//
//  LRFPersonality.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 2014-03-01.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

/*
 
 +----------------------------------+
 |									|
 |	=[happiness]=	=[chattiness]=	|
 |									|
 |	=[ humour  ]=	=[enthusiasm]=	|
 |									|
 |	=[neediness]=	=[ bravery  ]=	|
 |									|
 |									|
 |	 [ 10-pin connector is here]	|
 |									|
 |									|
 |	[ battery+ ]     [ battery+ ]	|
 |									|
 +----------------------------------+
 
 Ch	Adc	Name
 ----------------
 A	2	happiness
 B	3	chattiness
 C	1	humour
 D	4	enthusiasm
 E	0	neediness
 F	5	bravery
 
 */

#include "LRFPersonality.h"
#include "LRFEvents.h"
#include "LRFMemory.h"
#include "LRFHardware.h"
#include "LRFEyes.h"
#include "LRFSpeech.h"
#include "LRFExpressions.h"

#if LRF_EXPRESSIONS_DEBUG
#include "LRFDebug.h"
const char debug_title[] = "per";
void lrf_personality_print_archetype(LRFPersonalityProfile profile);
#endif



const LRFPersonalityStruct PROGMEM lrfPersonalityProfiles[LRF_PERSONALITY_PROFILE_COUNT] = {
	
	// b, ha, e, n, c, hu
	{3,3,3,3,3,3},	// custom
	{4,4,4,1,5,4},	// chatter
	{5,2,5,1,5,3},	// diva
	{4,2,1,5,1,1},	// plato
	{5,3,5,3,4,4},	// rascal
	{4,4,1,3,2,2},	// nanna
	{2,2,2,4,2,3},	// office
	{4,5,5,1,5,4},	// party
	{1,3,4,4,2,2},	// scaredy
	{3,5,4,2,4,5},	// jolly
	{5,1,4,2,3,2}	// evil

};

LRFPersonality lrfPersonality = {
	LRF_PERSONALITY_LEVEL_NORMAL,
	LRF_PERSONALITY_LEVEL_NORMAL,
	LRF_PERSONALITY_LEVEL_NORMAL,
	LRF_PERSONALITY_LEVEL_NORMAL,
	LRF_PERSONALITY_LEVEL_NORMAL,
	LRF_PERSONALITY_LEVEL_NORMAL
};


//------------------------------------------------------------
#pragma mark - Setup
//------------------------------------------------------------

void lrf_personality_setup(void)
{
	lrf_personality_select_on_startup();
	lrf_personality_has_read_shoelaces_check();
	
	// load the personality
}

//------------------------------------------------------------
#pragma mark - Selection
//------------------------------------------------------------

void lrf_personality_select_profile(LRFPersonalityProfile profile)
{
	if(profile == LRFPersonalityProfile_Custom)
	{
		lrf_personality_read_shoelaces();
		return;
	}
	
	// copy profile data from program space
	LRFPersonalityData data;
	data.raw = pgm_read_dword(&lrfPersonalityProfiles[profile]);
	
	// move traits into local memory
	lrfExpressions.signatureIdx	= profile;
	lrfPersonality.bravery		= data.profile.bravery;
	lrfPersonality.happiness	= data.profile.happiness;
	lrfPersonality.chattiness	= data.profile.chattiness;
	lrfPersonality.enthusiasm	= data.profile.enthusiasm;
	lrfPersonality.neediness	= data.profile.neediness;
	lrfPersonality.humour		= data.profile.humour;

	// save new trait configuration
	lrf_memory_save_personality();
}

void lrf_personality_selector(void)
{
	LRFPersonalityProfile profile = LRFPersonalityProfile_Custom;
	unsigned char touch = 0;
	unsigned long timestamp = 0;
	
#if LRF_PERSONALITY_DEBUG
	lrf_debug_tag(debug_title, "selector");
#endif
	
	lrf_expressions_load_and_block(LRFExpression_Alarm);
	delay(1000);
	
	while(true)
	{
		// speaking our signature sound
		lrfExpressions.signatureIdx = (unsigned char)profile;
		lrf_expressions_load_and_block(LRFExpression_Signature);
		delay(500);
		
#if LRF_PERSONALITY_SELECTOR_USE_TOUCH
		
		// wait for touch down (or serial input)
		timestamp = millis();
		while(touch==0 && (millis()-timestamp) < LRF_PERSONALITY_SELECTION_TOUCH_TIMEOUT*2)
		{
			lrf_touch_read(&touch);
			delay(10);
		}
		
		// check our duration timeout
		if(touch == 0)
		{
#if LRF_PERSONALITY_DEBUG
			lrf_debug_tag(debug_title, "timeout");
#endif
			lrf_expressions_load_and_block(LRFExpression_Pout);
			return;
		}
		
#if LRF_PERSONALITY_DEBUG
		lrf_debug_tag(debug_title, "touchdown");
#endif

		// mark the time touch went down
		timestamp = millis();
		
		// wait for touch up
		while(touch==1)
		{
			lrf_touch_read(&touch);
			delay(10);

			// measure our touch
			if((millis()-timestamp) > LRF_PERSONALITY_SELECTION_TOUCH_TIMEOUT)
			{
				LRF_RGB_PORT = LRF_RGB_RIGHT_RED | LRF_RGB_LEFT_RED;
				//lrf_eyes_set_color(LRFColor_SoftRed);
				delay(50);
				LRF_RGB_PORT = 0;
				//lrf_eyes_set_color(LRFColor_Blank);
				delay(200);
			}
		}
		
		// measure our touch
#if LRF_PERSONALITY_DEBUG
		lrf_debug_tag(debug_title, "touchup");
#endif
		
		// what should we do based on that?
		if((millis()-timestamp) < LRF_PERSONALITY_SELECTION_TOUCH_TIMEOUT)
		{
			profile = (LRFPersonalityProfile)((profile+1) % LRF_PERSONALITY_PROFILE_COUNT);
#if LRF_PERSONALITY_DEBUG
			lrf_debug_tag(debug_title, "next:", false);
			lrf_personality_print_archetype(profile);
			Serial.println();
#endif
		}
		else
		{
#if LRF_PERSONALITY_DEBUG
			lrf_debug_tag(debug_title, "select:");
			lrf_personality_print_archetype(profile);
			Serial.println();
#endif
			lrf_personality_select_profile((LRFPersonalityProfile)profile);
			break;
		}
#else
		
		while(!Serial.available()) delay(10);
		
		unsigned char r = Serial.read();
		
		if(r == 'r')
		{
#if LRF_PERSONALITY_DEBUG
			lrf_debug_tag(debug_title, "repeat:", false);
			lrf_personality_print_archetype(profile);
			Serial.println();
#endif
		}
		else if(r == 's')
		{
#if LRF_PERSONALITY_DEBUG
			lrf_debug_tag(debug_title, "select:", false);
			lrf_personality_print_archetype(profile);
			Serial.println();
#endif
			lrf_personality_select_profile((LRFPersonalityProfile)profile);
			break;
		}
		else
		{
			profile = (profile+1) % LRF_PERSONALITY_PROFILE_COUNT;
#if LRF_PERSONALITY_DEBUG
			lrf_debug_tag(debug_title, "next:", false);
			lrf_personality_print_archetype(profile);
			Serial.println();
#endif
		}

		
#endif
		
		
		delay(200);
	}
}

void lrf_personality_select_on_startup(void)
{
	unsigned char touch = 1;
	unsigned long timestamp = millis();
	bool flag = false;
	
	lrf_irda_read_disable();
	
	if(lrf_utils_is_holiday_hardware())
	{
		lrf_eyes_set_color(LRFColor_SoftGreen);
	}
	else
	{
		lrf_eyes_set_color(LRFColor_SoftRed);
	}
	delay(1000);
	
	
	while(touch == 1)
	{
		lrf_touch_read(&touch);
		delay(10);
		
		if(!flag && (millis()-timestamp) > LRF_PERSONALITY_SELECTION_STARTUP_TOUCH_TIME)
		{
			lrf_eyes_set_color(LRFColor_SoftBlue);
			flag = true;
		}
	}

	lrf_eyes_set_color(LRFColor_Blank);
	
	if((millis()-timestamp) > LRF_PERSONALITY_SELECTION_STARTUP_TOUCH_TIME)
	{
		lrf_personality_selector();
	}
}

//------------------------------------------------------------
#pragma mark - Reading Shoelaces
//------------------------------------------------------------

void lrf_personality_has_read_shoelaces_check(void)
{
	// check our first time memory address
	unsigned char value = lrf_memory_read(LRF_MEMORY_HAS_READ_SHOELACES_ADDR);
	
	// if we haven't read our shoelaces yet, then let's flash green and wait for input
	if(value != LRF_MEMORY_HAS_READ_SHOELACES_VALUE)
	{
		bool done = false;
		unsigned char touch = 0;
		unsigned long timestamp = 0;
		unsigned char color = 0;

#if LRF_PERSONALITY_DEBUG
		lrf_debug_tag(debug_title, "first");
#endif

#if LRF_PERSONALITY_SHOELACE_CHECK_AUTO
		lrf_personality_read_shoelaces();
#else
		// loop until we're ready to check shoelaces (needs to be done, can always be rescanned)
		while(!done)
		{
			
			// wait for touch down
			while(touch == 0)
			{
				lrf_touch_read(&touch);
				lrf_eyes_set_color((LRFColor)(LRFColor_SoftRed+color));
				color = (color+1) % 3;
				delay(100);
				lrf_eyes_set_color(LRFColor_Blank);
				delay(100);
			}
			
			timestamp = millis();
			lrf_eyes_set_color(LRFColor_SoftGreen);
			
			// wait for touch up
			while(touch == 1)
			{
				lrf_touch_read(&touch);
				delay(100);
				
				if((millis()-timestamp) > LRF_PERSONALITY_SELECTION_TOUCH_TIMEOUT)
				{
					lrf_eyes_set_color(LRFColor_SoftBlue);
				}
			}
			
			lrf_eyes_set_color(LRFColor_Blank);

			if((millis()-timestamp) > LRF_PERSONALITY_SELECTION_TOUCH_TIMEOUT)
			{
				lrf_personality_read_shoelaces();
				done = true;
			}
		}
#endif
	}
}

unsigned char lrf_personality_read_shoelace(unsigned char idx)
{
	unsigned int val = analogRead(idx);
	unsigned char res = 3;
	
	if(val >= 900)		res = 3;	// no resistor = is same as normal
	else if(val >= 700)	res = 1;	// 27K = VERY LOW
	else if(val >= 575)	res = 2;	// 15K = LOW
	else if(val >= 450)	res = 3;	// 9.1K = NORMAL
	else if(val >= 300)	res = 4;	// 4.7K = HIGH
	else if(val >= 150)	res = 5;	// 2.2K = VERY HIGH
	return res;
}

void lrf_personality_read_shoelaces(void)
{
#if LRF_PERSONALITY_DEBUG
	lrf_debug_tag(debug_title, "shoelaces");
#endif
	
	// get local copies of our six resistor values
	unsigned char values[6];
	for(unsigned char i=0; i<6; i++)
	{
		values[i] = lrf_personality_read_shoelace(i);
		delay(5);
	}
	
	// copy locals into our positions
	lrfPersonality.neediness	= values[LRF_PERSONALITY_NEEDINESS_CHANNEL];
	lrfPersonality.humour		= values[LRF_PERSONALITY_HUMOUR_CHANNEL];
	lrfPersonality.happiness	= values[LRF_PERSONALITY_HAPPINESS_CHANNEL];
	lrfPersonality.chattiness	= values[LRF_PERSONALITY_CHATTINESS_CHANNEL];
	lrfPersonality.enthusiasm	= values[LRF_PERSONALITY_ENTHUSIASM_CHANNEL];
	lrfPersonality.bravery		= values[LRF_PERSONALITY_BRAVERY_CHANNEL];
	
#if LRF_PERSONALITY_DEBUG
	lrf_personality_print();
#endif
	
	// set our 'read shoelaces' flag
	lrf_memory_write(LRF_MEMORY_HAS_READ_SHOELACES_ADDR, LRF_MEMORY_HAS_READ_SHOELACES_VALUE);
	
	// write our personality
	lrf_memory_save_personality();
}


//------------------------------------------------------------
#pragma mark - Debug
//------------------------------------------------------------

#if LRF_PERSONALITY_DEBUG

void lrf_personality_print(void)
{
#if LRF_PERSONALITY_DEBUG_VERBOSE
	Serial.print("neediness:");
	Serial.println(lrfPersonality.neediness,DEC);
	Serial.print("humour:");
	Serial.println(lrfPersonality.humour,DEC);
	Serial.print("happiness:");
	Serial.println(lrfPersonality.happiness,DEC);
	Serial.print("chattiness:");
	Serial.println(lrfPersonality.chattiness,DEC);
	Serial.print("enthusiasm:");
	Serial.println(lrfPersonality.enthusiasm,DEC);
	Serial.print("bravery:");
	Serial.println(lrfPersonality.bravery,DEC);
#else
	lrf_debug_list(debug_title, &lrfPersonality.neediness, 6);
#endif
}

void lrf_personality_print_archetype(LRFPersonalityProfile profile)
{
#if LRF_PERSONALITY_DEBUG_VERBOSE
	if(profile == 0)		Serial.print("custom");
	else if(profile == 1)	Serial.print("chatter");
	else if(profile == 2)	Serial.print("diva");
	else if(profile == 3)	Serial.print("plato");
	else if(profile == 4)	Serial.print("rascal");
	else if(profile == 5)	Serial.print("nanna");
	else if(profile == 6)	Serial.print("office");
	else if(profile == 7)	Serial.print("party");
	else if(profile == 8)	Serial.print("scaredy");
	else if(profile == 9)	Serial.print("jolly");
	else if(profile == 10)	Serial.print("evil");
	Serial.print("bot");
#else
	lrf_debug_tag(debug_title, "profile:", false);
	Serial.println(profile,DEC);
#endif
}
#endif
