//
//  LRFEyes.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 1/14/2014.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#include "LRFEyes.h"
#include "LRFHardware.h"
#include "LRFPower.h"

//-------------------------------------------------------------
#pragma mark - Variables
//-------------------------------------------------------------

LRFEyes lrfEyes = { LRFEyesState_Idle };

unsigned char lrfDelayCounter = 0;

#if LRF_EYES_DEBUG
void lrf_eyes_debug_color_value(LRFEyeColorValue *val)
{
	Serial.print("[");
	Serial.print(val->red,DEC);
	Serial.print(":");
	Serial.print(val->green,DEC);
	Serial.print(":");
	Serial.print(val->blue,DEC);
	Serial.print("]");
}

void lrf_rgb_debug_duty_map(void)
{
	for(unsigned char i=0; i<LRF_RGB_VALUE_MAX; i++)
	{
		Serial.print(i,DEC);
		Serial.print(": ");
		Serial.println(lrf_rgb_duty_map[i],BIN);
	}
}
#endif


//-------------------------------------------------------------
#pragma mark - Reset
//-------------------------------------------------------------

void lrf_eyes_reset(void)
{
	lrfEyes.state = LRFEyesState_Idle;
	lrf_rgb_set(LRFRgbMask_All,0);
}

void lrf_eyes_reset_pattern(void)
{
	// if we've got a flip pattern, let's flip start and target colors
	if(lrfEyes.pattern.mode == LRFPatternMode_FlatFlip ||
	   lrfEyes.pattern.mode == LRFPatternMode_FlatFlipToggle ||
	   lrfEyes.pattern.mode == LRFPatternMode_FadeFlip ||
	   lrfEyes.pattern.mode == LRFPatternMode_FadeFlipToggle)
	{
		lrfEyes.currentLeft = lrfEyes.startingLeft;
		lrfEyes.currentRight = lrfEyes.startingRight;
		lrfEyes.startingLeft = lrfEyes.targetLeft;
		lrfEyes.startingRight = lrfEyes.targetRight;
		lrfEyes.targetLeft = lrfEyes.currentLeft;
		lrfEyes.targetRight = lrfEyes.currentRight;
	}
	// if we've got a random pattern, let's fetch random colors
	else if(lrfEyes.pattern.mode == LRFPatternMode_RandomFlat ||
			lrfEyes.pattern.mode == LRFPatternMode_RandomFade ||
			lrfEyes.pattern.mode == LRFPatternMode_RandomBoomerang)
	{
		lrf_utils_color_lookup_random(&lrfEyes.startingLeft);
		lrf_utils_color_lookup_random(&lrfEyes.targetLeft);
		lrfEyes.startingRight = lrfEyes.startingLeft;
		lrfEyes.targetRight = lrfEyes.targetLeft;
		lrfEyes.currentLeft = lrfEyes.startingLeft;
		lrfEyes.currentRight = lrfEyes.startingRight;
	}
	// random exception for toggle values
	else if(lrfEyes.pattern.mode == LRFPatternMode_RandomToggle)
	{
		lrf_utils_color_lookup_random(&lrfEyes.startingLeft);
		lrf_utils_color_lookup_random(&lrfEyes.startingRight);
		lrfEyes.targetLeft = lrfEyes.startingRight;
		lrfEyes.targetRight = lrfEyes.startingLeft;
		lrfEyes.currentLeft = lrfEyes.startingLeft;
		lrfEyes.currentRight = lrfEyes.startingRight;
	}
	else
	{
		lrfEyes.currentLeft = lrfEyes.startingLeft;
		lrfEyes.currentRight = lrfEyes.startingRight;
	}
	
	// update our state so we begin on next process call
	lrfEyes.state = LRFEyesState_Ready;
}

//-------------------------------------------------------------
#pragma mark - Personality
//-------------------------------------------------------------

void lrf_eyes_apply_personality(void)
{
	
}

//-------------------------------------------------------------
#pragma mark - Setting & Playing
//-------------------------------------------------------------

void lrf_eyes_update(void)
{
	if(lrfPower.isHungry)
	{
		// this is a pretty heavy function
		lrf_rgb_set(LRFRgbMask_LeftBlue, lrfEyes.currentLeft.blue >> 3);
		lrf_rgb_set(LRFRgbMask_LeftGreen, lrfEyes.currentLeft.green >> 3);
		lrf_rgb_set(LRFRgbMask_LeftRed, lrfEyes.currentLeft.red >> 3);
		lrf_rgb_set(LRFRgbMask_RightBlue, lrfEyes.currentRight.blue >> 3);
		lrf_rgb_set(LRFRgbMask_RightGreen, lrfEyes.currentRight.green >> 3);
		lrf_rgb_set(LRFRgbMask_RightRed, lrfEyes.currentRight.red >> 3);
	}
	else
	{
		lrf_rgb_set(LRFRgbMask_LeftBlue, lrfEyes.currentLeft.blue);
		lrf_rgb_set(LRFRgbMask_LeftGreen, lrfEyes.currentLeft.green);
		lrf_rgb_set(LRFRgbMask_LeftRed, lrfEyes.currentLeft.red);
		lrf_rgb_set(LRFRgbMask_RightBlue, lrfEyes.currentRight.blue);
		lrf_rgb_set(LRFRgbMask_RightGreen, lrfEyes.currentRight.green);
		lrf_rgb_set(LRFRgbMask_RightRed, lrfEyes.currentRight.red);
	}
#if LRF_EYES_DEBUG_VERBOSE
	lrf_rgb_debug_duty_map();
#endif
}

void lrf_eyes_set_color(LRFColor color)
{
	lrf_utils_color_lookup(color, &lrfEyes.currentLeft);
	lrf_utils_color_lookup(color, &lrfEyes.currentRight);
	lrf_eyes_update();
}

void lrf_eyes_set_colors(LRFColor left, LRFColor right)
{
	lrf_utils_color_lookup(left, &lrfEyes.currentLeft);
	lrf_utils_color_lookup(right, &lrfEyes.currentRight);
	lrf_eyes_update();
}

void lrf_eyes_play_pattern_and_block(LRFPattern pattern, bool applyPersonality)
{
	lrf_eyes_play_pattern_duration_and_block(pattern, 0, applyPersonality);
}

void lrf_eyes_play_pattern_duration_and_block(LRFPattern pattern, unsigned int duration, bool applyPersonality)
{
	lrf_eyes_play_pattern(pattern, applyPersonality);

	if(duration != 0) lrfEyes.duration = duration;
	
	while(lrfEyes.state != LRFEyesState_Done)
	{
		lrf_eyes_process();
		delay(5);
	}
	
	lrf_eyes_reset();
}

void lrf_eyes_play_pattern(LRFPattern pattern, bool applyPersonality)
{
	// copy the pattern structure from the unified obj
	lrfEyes.pattern = pattern.pattern;
	
	// lookup our colors (flip values for toggles)
	if(lrfEyes.pattern.mode == LRFPatternMode_FlatFlipToggle ||
	   lrfEyes.pattern.mode == LRFPatternMode_FadeToggle ||
	   lrfEyes.pattern.mode == LRFPatternMode_FadeFlipToggle ||
	   lrfEyes.pattern.mode == LRFPatternMode_BoomerangToggle)
	{
		lrf_utils_color_lookup(lrfEyes.pattern.start, &lrfEyes.startingLeft);
		lrf_utils_color_lookup(lrfEyes.pattern.target, &lrfEyes.startingRight);
		lrf_utils_color_lookup(lrfEyes.pattern.target, &lrfEyes.targetLeft);
		lrf_utils_color_lookup(lrfEyes.pattern.start, &lrfEyes.targetRight);
	}
	else
	{
		lrf_utils_color_lookup(lrfEyes.pattern.start, &lrfEyes.startingLeft);
		lrf_utils_color_lookup(lrfEyes.pattern.start, &lrfEyes.startingRight);
		lrf_utils_color_lookup(lrfEyes.pattern.target, &lrfEyes.targetLeft);
		lrf_utils_color_lookup(lrfEyes.pattern.target, &lrfEyes.targetRight);
	}

	// prefill a standard duration?
	lrfEyes.duration = lrf_utils_duration_lookup(LRFDuration_Long);
	
	// shift our values based on personality
	if(applyPersonality) lrf_eyes_apply_personality();

	// prep the pattern
	lrf_eyes_reset_pattern();
}


//-------------------------------------------------------------
#pragma mark - Process
//-------------------------------------------------------------

inline void lrf_eyes_process_value(unsigned char *currentValue, unsigned char *startingValue, unsigned char *targetValue, float *factor)
{
	signed char tempVal = *targetValue - *startingValue;
	float tempFloat = (tempVal * *factor);
	*currentValue = *startingValue + ((signed char)tempFloat);
}

void lrf_eyes_process(void)
{
	if(lrfEyes.state == LRFEyesState_Idle || lrfEyes.state == LRFEyesState_Done)
	{
		return;
	}
	
	unsigned long now = millis();
	
	if(lrfEyes.state == LRFEyesState_Ready)
	{
		// mark the start time
		lrfEyes.startTime = now;
		
		// load the current color into leds
		lrf_eyes_update();
		
		// update the state
		lrfEyes.state = LRFEyesState_Running;
	}
	else if(lrfEyes.state == LRFEyesState_Running)
	{
		// have we completed our pattern?
		if(now > lrfEyes.startTime + lrfEyes.duration)
		{
			lrfEyes.state = LRFEyesState_Done;
			
			// note: maybe we can use a fadeout state?
		}
		else
		{
			// let's calculate our values
			float factor;
			
			// flat patterns do nothing
			if(lrfEyes.pattern.mode <= LRFPatternMode_FlatFlipToggle ||
			   lrfEyes.pattern.mode == LRFPatternMode_RandomFlat)
			{
				return;
			}
			// fading patterns move from A to B
			else if(lrfEyes.pattern.mode <= LRFPatternMode_FadeFlipToggle ||
					lrfEyes.pattern.mode == LRFPatternMode_RandomFade)
			{
				factor = (float)(now - lrfEyes.startTime) / ((float)lrfEyes.duration);
			}
			// boomerang patterns move from A to B to A
			else if(lrfEyes.pattern.mode <= LRFPatternMode_BoomerangToggle ||
					lrfEyes.pattern.mode == LRFPatternMode_RandomBoomerang)
			{
				factor = (float)(now - lrfEyes.startTime) / ((float)lrfEyes.duration/2);
				if(factor > 1.0) factor = 2.0 - factor;
			}
			
#if LRF_EYES_DEBUG
			Serial.print("f: ");
			Serial.println(factor);
#endif
			lrf_eyes_process_value(&lrfEyes.currentLeft.red,
								   &lrfEyes.startingLeft.red,
								   &lrfEyes.targetLeft.red,
								   &factor);
			lrf_eyes_process_value(&lrfEyes.currentLeft.green,
								   &lrfEyes.startingLeft.green,
								   &lrfEyes.targetLeft.green,
								   &factor);
			lrf_eyes_process_value(&lrfEyes.currentLeft.blue,
								   &lrfEyes.startingLeft.blue,
								   &lrfEyes.targetLeft.blue,
								   &factor);
			lrf_eyes_process_value(&lrfEyes.currentRight.red,
								   &lrfEyes.startingRight.red,
								   &lrfEyes.targetRight.red,
								   &factor);
			lrf_eyes_process_value(&lrfEyes.currentRight.green,
								   &lrfEyes.startingRight.green,
								   &lrfEyes.targetRight.green,
								   &factor);
			lrf_eyes_process_value(&lrfEyes.currentRight.blue,
								   &lrfEyes.startingRight.blue,
								   &lrfEyes.targetRight.blue,
								   &factor);
		}
		
		lrf_eyes_update();
	}
	
}

void lrf_eyes_fadeout(void)
{
	bool done = false;
	
	while(!done)
	{
		done = true;
		
		if(lrfEyes.currentLeft.red > 0)		{ lrfEyes.currentLeft.red--;	done=false; }
		if(lrfEyes.currentLeft.green > 0)	{ lrfEyes.currentLeft.green--;	done=false; }
		if(lrfEyes.currentLeft.blue > 0)	{ lrfEyes.currentLeft.blue--;	done=false; }

		if(lrfEyes.currentRight.red > 0)	{ lrfEyes.currentRight.red--;	done=false; }
		if(lrfEyes.currentRight.green > 0)	{ lrfEyes.currentRight.green--; done=false; }
		if(lrfEyes.currentRight.blue > 0)	{ lrfEyes.currentRight.blue--;	done=false; }

		lrf_eyes_update();
		delay(1);
	}
}


//-------------------------------------------------------------
#pragma mark - Depricated
//-------------------------------------------------------------


//void lrf_eyes_load_color(LRFEyeColor left, LRFEyeColor right)
//{
//	// this is a pretty heavy function
//	lrf_rgb_set(LRFRgbMask_LeftBlue, lrfColorValues[left].blue);
//	lrf_rgb_set(LRFRgbMask_LeftGreen, lrfColorValues[left].green);
//	lrf_rgb_set(LRFRgbMask_LeftRed, lrfColorValues[left].red);
//	lrf_rgb_set(LRFRgbMask_RightBlue, lrfColorValues[right].blue);
//	lrf_rgb_set(LRFRgbMask_RightGreen, lrfColorValues[right].green);
//	lrf_rgb_set(LRFRgbMask_RightRed, lrfColorValues[right].red);
//#if LRF_EYES_DEBUG_VERBOSE
//	lrf_rgb_debug_duty_map();
//#endif
//}

//void lrf_eyes_load_color_value(LRFEyeColorValue *left, LRFEyeColorValue *right)
//{
//	// this is a pretty heavy function
//	lrf_rgb_set(LRFRgbMask_LeftBlue, left->blue);
//	lrf_rgb_set(LRFRgbMask_LeftGreen, left->green);
//	lrf_rgb_set(LRFRgbMask_LeftRed, left->red);
//	lrf_rgb_set(LRFRgbMask_RightBlue, right->blue);
//	lrf_rgb_set(LRFRgbMask_RightGreen, right->green);
//	lrf_rgb_set(LRFRgbMask_RightRed, right->red);
//#if LRF_EYES_DEBUG_VERBOSE
//	lrf_rgb_debug_duty_map();
//#endif
//}

//void lrf_eyes_load_expression(LRFEyesExpressionName name)
//{
//	if(name >= LRF_EYES_MAX_EXPRESSIONS) return;
//	
//	// load the expression in our eyes object
//	lrfEyes.expression = lrfEyesExpressions[name];
//	
//	// reset the repetition counter
//	lrfEyes.repeatCounter = 0;
//	
//	// unpack the color data
//	lrfEyes.startingLeftColor = lrfColorValues[lrfEyes.expression->startingColors.left];
//	lrfEyes.startingRightColor = lrfColorValues[lrfEyes.expression->startingColors.right];
//	lrfEyes.currentLeftColor = lrfEyes.startingLeftColor;
//	lrfEyes.currentRightColor = lrfEyes.startingRightColor;
//	lrfEyes.targetLeftColor = lrfColorValues[lrfEyes.expression->targetColors.left];
//	lrfEyes.targetRightColor = lrfColorValues[lrfEyes.expression->targetColors.right];
//	
//#if LRF_EYES_DEBUG
//	Serial.print("sc: ");
//	Serial.print(expression->startingColors.left,DEC);
//	Serial.print("-");
//	Serial.print(expression->startingColors.right,DEC);
//	Serial.print(" ");
//	lrf_eyes_debug_color_value(&lrfEyes.startingLeftColor);
//	Serial.print(" ");
//	lrf_eyes_debug_color_value(&lrfEyes.startingRightColor);
//	Serial.println();
//	
//	Serial.print("tc: ");
//	Serial.print(expression->targetColors.left,DEC);
//	Serial.print("-");
//	Serial.print(expression->targetColors.right,DEC);
//	Serial.print(" ");
//	lrf_eyes_debug_color_value(&lrfEyes.targetLeftColor);
//	Serial.print(" ");
//	lrf_eyes_debug_color_value(&lrfEyes.targetRightColor);
//	Serial.println();
//#endif
//	
//	// mark the starting time & ending time
//	lrfEyes.startTime = millis();
//	lrfEyes.endTime = lrfEyes.startTime + (lrfEyes.expression->duration * LRF_EYES_DURATION_SCALER);
//#if LRF_EYES_DEBUG
//	Serial.print("st: ");
//	Serial.print(lrfEyes.startTime,DEC);
//	Serial.print(" - et: ");
//	Serial.print(lrfEyes.endTime,DEC);
//	Serial.println();
//#endif
//	
//	// set the starting color
//	lrf_eyes_load_color_value(&lrfEyes.startingLeftColor, &lrfEyes.startingRightColor);
//	
//	// update the state
//	lrfEyes.state = LRFEyesState_Running;
//}


//void lrf_eyes_process()
//{
//	// return quickly if we're not doing anything
//	if(lrfEyes.state == LRFEyesState_Idle) return;
//
//	lrfDelayCounter++;
//	if(lrfDelayCounter < 0xf0) return;
//	lrfDelayCounter = 0;
//	
//	// if we're done then we should fade out
//	if (lrfEyes.state == LRFEyesState_Done)
//	{
//		lrfDelayCounter = 0xf0;
////		lrfDelayCounter = 0xcf;
//		
//		// decrement all colors
//		bool done=true;
//		if(lrfEyes.currentLeftColor.red > 0)
//		{
//			lrfEyes.currentLeftColor.red--;
//			done=false;
//		}
//		if(lrfEyes.currentLeftColor.green > 0)
//		{
//			lrfEyes.currentLeftColor.green--;
//			done=false;
//		}
//		if(lrfEyes.currentLeftColor.blue > 0)
//		{
//			lrfEyes.currentLeftColor.blue--;
//			done=false;
//		}
//		if(lrfEyes.currentRightColor.red > 0)
//		{
//			lrfEyes.currentRightColor.red--;
//			done=false;
//		}
//		if(lrfEyes.currentRightColor.green > 0)
//		{
//			lrfEyes.currentRightColor.green--;
//			done=false;
//		}
//		if(lrfEyes.currentRightColor.blue > 0)
//		{
//			lrfEyes.currentRightColor.blue--;
//			done=false;
//		}
//		
//		// load the adjusted color
//		lrf_eyes_load_color_value(&lrfEyes.currentLeftColor, &lrfEyes.currentRightColor);
//		
//		// if we're done, change state
//		if(done)
//		{
//			lrfEyes.state = LRFEyesState_Idle;
//#if LRF_EYES_DEBUG
//			Serial.println("state: i");
//#endif
//		}
//	}
//	
//	// otherwise our rgb process is still running
//	else
//	{
//		// are we beyond our time?
//		if(millis() > lrfEyes.endTime)
//		{
//			if(lrfEyes.expression->repetitions == 0 ||
//			   lrfEyes.repeatCounter == (lrfEyes.expression->repetitions-1))
//			{
//				lrfEyes.state = LRFEyesState_Done;
//#if LRF_EYES_DEBUG
//				Serial.println("state: d");
//#endif
//			}
//			else
//			{
//				if(lrfEyes.expression->toggle == 0)
//				{
//					// increment the repeat counter
//					lrfEyes.repeatCounter++;
//
//					// load the starting color into the current color
//					lrfEyes.currentLeftColor = lrfEyes.startingLeftColor;
//					lrfEyes.currentRightColor = lrfEyes.startingRightColor;
//				}
//				else
//				{
//					if(lrfEyes.state == LRFEyesState_Toggling)
//					{
//						// increment the repeat counter
//						lrfEyes.repeatCounter++;
//
//						// update the state
//						lrfEyes.state = LRFEyesState_Running;
//					}
//					else
//					{
//						// update the state
//						lrfEyes.state = LRFEyesState_Toggling;
//					}
//				}
//
//				// mark the starting time & ending time
//				lrfEyes.startTime = millis();
//				lrfEyes.endTime = lrfEyes.startTime + (lrfEyes.expression->duration * LRF_EYES_DURATION_SCALER);
//			}
//			
//		}
//		else
//		{
//			// let's calculate our values
//			float factor = (float)(millis() - lrfEyes.startTime) / ((float)lrfEyes.expression->duration * LRF_EYES_DURATION_SCALER);
//			if(lrfEyes.state == LRFEyesState_Toggling) factor = 1.0-factor;
//#if LRF_EYES_DEBUG
//			Serial.print("f: ");
//			Serial.println(factor);
//#endif
//			lrf_eyes_process_value(&lrfEyes.currentLeftColor.red,
//								   &lrfEyes.startingLeftColor.red,
//								   &lrfEyes.targetLeftColor.red,
//								   &factor);
//			lrf_eyes_process_value(&lrfEyes.currentLeftColor.green,
//								   &lrfEyes.startingLeftColor.green,
//								   &lrfEyes.targetLeftColor.green,
//								   &factor);
//			lrf_eyes_process_value(&lrfEyes.currentLeftColor.blue,
//								   &lrfEyes.startingLeftColor.blue,
//								   &lrfEyes.targetLeftColor.blue,
//								   &factor);
//			lrf_eyes_process_value(&lrfEyes.currentRightColor.red,
//								   &lrfEyes.startingRightColor.red,
//								   &lrfEyes.targetRightColor.red,
//								   &factor);
//			lrf_eyes_process_value(&lrfEyes.currentRightColor.green,
//								   &lrfEyes.startingRightColor.green,
//								   &lrfEyes.targetRightColor.green,
//								   &factor);
//			lrf_eyes_process_value(&lrfEyes.currentRightColor.blue,
//								   &lrfEyes.startingRightColor.blue,
//								   &lrfEyes.targetRightColor.blue,
//								   &factor);
//			lrf_eyes_load_color_value(&lrfEyes.currentLeftColor, &lrfEyes.currentRightColor);
//		}
//	}
//	
//}




