//
//  LRFExpressions.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 2014-03-14.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#include "LRFExpressions.h"
#include "LRFSpeech.h"
#include "LRFEyes.h"
#include "LRFHardware.h"
#include "LRFSensors.h"
#include "LRFEvents.h"
#include "LRFPersonality.h"
#include "LRFSongData.h"

#if LRF_EXPRESSIONS_DEBUG
#include "LRFDebug.h"
const char debug_title[] = "exp";
#endif

LRFExpressions lrfExpressions = {
	LRFExpressionState_Idle,
	LRFExpression_None,
	LRFExpression_None,
	(LRFFunctionPtr)0
};

//LRFPattern emptyPattern = { .raw = 0 };
//LRFSound emptySound = { .raw = 0 };

LRFPattern emptyPattern;
LRFSound emptySound;

//------------------------------------------------------------
#pragma mark - Done & Begin Placeholders
//------------------------------------------------------------

void lrf_expressions_setup(void)
{
	emptyPattern.raw = (int)0;
	emptySound.raw = (int)0;
}

void lrf_expressions_begin(void)
{
	// disable ir
	lrf_irda_read_disable();
	
	// disable the sensors
	lrf_sensors_disable();
	
	// disable the events
	lrf_events_disable();
	
	// update the state and reload our timers
	lrfExpressions.state = LRFExpressionState_Running;
	lrfExpressions.eyesTimer = 0;
}

void lrf_expressions_finish(void)
{
	// enable the ir
	lrf_irda_read_enable();
	
	// enable the sensors
	lrf_sensors_enable();
	
	// re-enable events
	lrf_events_enable();
	
	// callback function
	if(lrfExpressions.callback != 0)
	{
		LRFFunctionPtr callback = lrfExpressions.callback;
		lrfExpressions.callback = 0;
#if LRF_EXPRESSIONS_CALLBACK_DEBUG
		Serial.print("cb: 0x");
		Serial.println((long)callback,HEX);
#endif
		callback();
	}
#if LRF_EXPRESSIONS_CALLBACK_DEBUG
	else
	{
		Serial.println("cb: 0");
	}
#endif
}


//------------------------------------------------------------
#pragma mark - Singing
//------------------------------------------------------------

void lrf_expressions_sing(unsigned char startIdx, unsigned char endIdx)
{
#if LRF_EXPRESSIONS_DEBUG
	lrf_debug_tag(debug_title, "sing");
#endif

	// call beginning function
	lrf_expressions_begin();
	
	unsigned char idx = (startIdx == 0xff) ? 0 : startIdx;
	unsigned char end = (endIdx == 0xff) ? LRF_SONG_SIZE : endIdx;
	LRFSound sound;
	
	while(idx < end)
	{
		sound.raw = pgm_read_word(&lrfSong[idx]);

		lrf_eyes_set_color((LRFColor)(1 + idx%8));
		lrf_speech_play_sound_and_block(sound, false);
		
		idx++;
	}
	
	lrf_expressions_done();
}

//------------------------------------------------------------
#pragma mark - Chatter
//------------------------------------------------------------

void lrf_expressions_chatter(void)
{
#if LRF_EXPRESSIONS_DEBUG
	lrf_debug_tag(debug_title, "chat");
#endif
	
	// call beginning function
	lrf_expressions_begin();
	
	// small delay before a response
	delay(1000);
	
	//	unsigned char words = 2+(rand()%5);
	unsigned char words = 1 + lrfPersonality.chattiness + (rand() % lrfPersonality.chattiness);
	LRFSound sound;
	while(words--)
	{
		LRFColor left = (LRFColor)(1 + rand()%8);
		LRFColor right = (LRFColor)(1 + rand()%8);
		lrf_eyes_set_colors(left, right);
		
		sound.sound.note = (LRFNote)(rand()%(6+lrfPersonality.enthusiasm));
		sound.sound.intonation = (LRFIntonation)(rand()%5);
		sound.sound.duration = (LRFDuration)(6 - lrfPersonality.enthusiasm + rand()%3);
		sound.sound.pause = (LRFDuration)(5 - lrfPersonality.enthusiasm + rand()%3);
		sound.sound.octave = (LRFOctave)(1 + rand()%lrfPersonality.enthusiasm);
		
		lrf_speech_play_sound_and_block(sound);
	}
	lrf_eyes_set_color(LRFColor_Blank);
	
	// done!
	lrf_expressions_done();
}


//------------------------------------------------------------
#pragma mark - Loading
//------------------------------------------------------------

void lrf_expressions_load_pattern(LRFPattern pattern)
{
	if(pattern.raw == emptyPattern.raw)
	{
		// load the pattern from program memory
		if(lrfExpressions.current == LRFExpression_Signature)
		{
			pattern.raw = pgm_read_word(&lrfSignaturePatterns[lrfExpressions.signatureIdx]);
		}
		else
		{
			pattern.raw = pgm_read_word(&lrfExpressionPatterns[lrfExpressions.current]);
		}
	}
	
	// play the patter
	lrf_eyes_play_pattern(pattern);
}

void lrf_expressions_load_sound(LRFSound sound)
{
	if(sound.raw == emptySound.raw)
	{
		// calculate the data address index & load the sound from program memory
		unsigned char idx;
		if(lrfExpressions.current == LRFExpression_Signature)
		{
			idx = (lrfExpressions.signatureIdx * LRF_SIGNATURE_SOUND_COUNT) + lrfExpressions.soundIdx;
			sound.raw = pgm_read_word(&lrfSignatureSounds[idx]);
		}
		else
		{
			idx = (lrfExpressions.current * LRF_EXPRESSIONS_SOUND_COUNT) + lrfExpressions.soundIdx;
			sound.raw = pgm_read_word(&lrfExpressionSounds[idx]);
		}
	}
	
	// play the sound (with a couple expression exemptions)
	if(lrfExpressions.current == LRFExpression_Snoring ||
	   lrfExpressions.current == LRFExpression_Signature ||
	   lrfExpressions.current == LRFExpression_Custom)
	{
		lrf_speech_play_sound(sound,false);
	}
	else
	{
		lrf_speech_play_sound(sound);
	}
}

void lrf_expressions_load(LRFExpression expression, LRFFunctionPtr callback)
{
	// save the name
	lrfExpressions.current = expression;
	lrfExpressions.callback = callback;
	
	// load the eyes pattern
	lrf_expressions_load_pattern(emptyPattern);
	
	// reset our sound index
	lrfExpressions.soundIdx = 0;
	lrfExpressions.soundCount = LRF_EXPRESSIONS_SOUND_COUNT;
	
	// load the first sound
	lrf_expressions_load_sound(emptySound);
	
	// copy the duration from sound into pattern
	lrfEyes.duration = lrfSpeech.duration;
	
	// call the begin function
	lrf_expressions_begin();
}

void lrf_expressions_load_pattern_and_sounds(LRFPattern pattern, LRFSound *sounds, char soundCount)
{
	lrfExpressions.current = LRFExpression_Custom;
	lrfExpressions.callback = 0;
	
	// load the eyes pattern
	lrf_expressions_load_pattern(pattern);
	
	// reset our sound index
	lrfExpressions.soundIdx = 0;
	lrfExpressions.soundCount = soundCount;
	
	// load the first sound
	lrf_expressions_load_sound(sounds[0]);
	
	// copy the duration from sound into pattern
	lrfEyes.duration = lrfSpeech.duration;
	
	// call the begin function
	lrf_expressions_begin();
}

void lrf_expressions_load_and_block(LRFExpression expression, LRFFunctionPtr callback)
{
	lrf_expressions_load(expression, callback);
	
	while(lrfExpressions.state != LRFExpressionState_Done)
	{
		lrf_expressions_process();
	}
}

void lrf_expressions_load_pattern_and_sounds_and_block(LRFPattern pattern, LRFSound *sounds, char soundCount)
{
	lrf_expressions_load_pattern_and_sounds(pattern, sounds, soundCount);
	
	while(lrfExpressions.state != LRFExpressionState_Done)
	{
		lrf_expressions_process(sounds);
	}
}


//------------------------------------------------------------
#pragma mark - Reset & Done
//------------------------------------------------------------

void lrf_expressions_done(void)
{
	if(lrfExpressions.next != LRFExpression_None)
	{
		lrf_expressions_load(lrfExpressions.next, lrfExpressions.callback);
		lrfExpressions.next = LRFExpression_None;
	}
	else
	{
		// fade out the eyes
		lrf_eyes_fadeout();
		
		// reset speech and eyes
		lrf_speech_reset();
		lrf_eyes_reset();
		
		lrfExpressions.state = LRFExpressionState_Done;

		// call the done function
		// lrfExpressions.doneFunction()
		lrf_expressions_finish();
	}
}


//------------------------------------------------------------
#pragma mark - Process
//------------------------------------------------------------

void lrf_expressions_process(LRFSound *sounds)
{
	if(lrfExpressions.state == LRFExpressionState_Running)
	{
		// check to see if our sound is done (sound controls everything else)
		if(lrfSpeech.state == LRFSpeechState_Done)
		{
			// let's increment our word
			lrfExpressions.soundIdx++;
			
			// check to see if we're done
			if(lrfExpressions.soundIdx == lrfExpressions.soundCount)
			{
				lrf_expressions_done();
				return;
			}
			
			// load the next sound (at soundIdx)
			if(sounds == 0)
			{
				lrf_expressions_load_sound(emptySound);
			}
			else
			{
				lrf_expressions_load_sound(sounds[lrfExpressions.soundIdx]);
			}
			
			// check for a null sound
			if(lrfSpeech.sound.note == LRFNote_NULL)
			{
				lrf_expressions_done();
				return;
			}
			
			// reload the eyes? or is it dynamically updated?
			lrf_eyes_reset_pattern();
		}
	
		// perform our processes
		lrf_speech_process();

		if(lrfExpressions.eyesTimer == 0)
		{
			lrf_eyes_process();
			lrfExpressions.eyesTimer = LRF_EYES_TIMER_RELOAD;
		}
		lrfExpressions.eyesTimer--;
	}
}



