//
//  LRFEventHandlers.h
//  LittleRobotFriends
//
//  Created by Mark Argo on 2014-03-14.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

/*!
 
 \file LRFEventHandlers.h
 \brief Contains all the built-in handlers that control how the LRF responds to different events.
 
 */

#ifndef __LRFEventHandlers_h
#define __LRFEventHandlers_h

#include <Arduino.h>

#define LRF_EVENT_HANDLERS_DEBUG			0

#define LRF_EVENTS_INTERACTION_START_IDX	0
#define LRF_EVENTS_INTERACTION_END_IDX		9
#define LRF_EVENTS_SYSTEM_START_IDX			10
#define LRF_EVENTS_SYSTEM_END_IDX			15

//! \enum	LRFEvent
//! \brief	Constants for different interaction and system events
//! \ingroup events
/*! The LRF current features 16 unique events broken down into interaction and system events. Interaction events include things like touch events, light sensor events, mic events and IR events. System events include things like sleep, wake, powerup and boredom. */

typedef enum
{
	//! @{ \name Interaction Events
	LRFEvent_Tap = 0,				//!< Single touch
	LRFEvent_Tickle = 1,			//!< Multiple touch
	LRFEvent_Hug = 2,				//!< Long touch
	LRFEvent_LightsOff = 3,			//!< Any level to dark
	LRFEvent_LightsOn = 4,			//!< Dark to any level
	LRFEvent_LightsBright = 5,		//!< Any level to bright
	LRFEvent_LightsLongDark = 6,	//!< Prolonged darkness
	LRFEvent_HeardLoudNoise = 7,	//!< Mic gets triggered
	LRFEvent_MessageReceived = 8,	//!< Irda received a message
	LRFEvent_MessageFailed = 9,		//!< Irda received a garbled message
	//! @}
	
	//! @{ \name System Events
	LRFEvent_IsHungry = 10,			//!< Battery is getting low
	LRFEvent_IsBored = 11,			//!< No interaction in a while
	LRFEvent_Sleep = 12,			//!< About to go to sleep
	LRFEvent_Wake = 13,				//!< Just woke up
	LRFEvent_PowerUp = 14,			//!< Just powered on (batteries in)
	LRFEvent_MoodChanged = 15,		//!< Depricated (don't use)
	//! @}
	
	LRFEvent_Unknown = 0xff			//!< Unrecognized event
} LRFEvent;

//! \typedef	LRFEventHandler
//! \brief		Function pointer with no parameters and no return value
//! \ingroup	events
//! \see		LittleRobotFriend::setEventHandler(LRFEvent event, LRFEventHandler handler)
/*! This function pointer type is compatible with how the LRF handling events. Used in conjunction with a setEventHandler function, this allows you to create a custom handler for events.
 
	Usage:
 
		void myEventHandler(void){
			lrf.doSomething(); // do something
		}
 
		lrf.setEventHandler(LRFEvent_Tickle, &myEventHandler);
 */
typedef void (*LRFEventHandler)(void);

extern LRFEventHandler lrfEventHandlers[];

#endif
