//
//  LRFUtils.h
//  LittleRobotFriends
//
//  Created by Mark Argo on 2014-05-01.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

/*!
 
 \file LRFUtils.h
 \brief Collection of custom types and handy functions used throughout the LRF codebase.
 
 */

#ifndef __LRFUtils_h
#define __LRFUtils_h

#include <Arduino.h>

//-------------------------------------------------------------
#pragma mark - Pattern Types
//-------------------------------------------------------------

// LRFColor - lookup index for color constants (4 bits max)
//! \enum	LRFColor
//! \brief	Constants for looking up color values stored in program memory (PROGMEM)
//! \ingroup light
/*! The LRF uses 16 predefined colors for all its light patterns. These values (LRFColorValue) can be viewed in the implementation of this file (LRFUtils.cpp). These values can be changed, but be sure to update any references you might make to your new custom color, as they may affect built in patterns in the LRFExpressionData.cpp or LRFSignatureData.cpp files. */

typedef enum
{
	LRFColor_Blank	= 0,	//!< LEDs off (no color)
	LRFColor_Red,			//!< Red
	LRFColor_Green,			//!< Green
	LRFColor_Blue,			//!< Blue
	LRFColor_Orange,		//!< Orange
	LRFColor_Cyan,			//!< Cyan
	LRFColor_Purple,		//!< Purple
	LRFColor_Yellow,		//!< Yellow
	LRFColor_Pink,			//!< Pink
	LRFColor_SoftRed,		//!< Soft Red
	LRFColor_SoftGreen,		//!< Soft Green
	LRFColor_SoftBlue,		//!< Soft Blue
	LRFColor_SoftYellow,	//!< Soft Yellow
	LRFColor_SoftCyan,		//!< Soft Cyan
	LRFColor_SoftPink,		//!< Soft Pink
	LRFColor_White = 15		//!< White
} __attribute__ ((packed)) LRFColor;

//! \struct LRFColorValue
//! \brief Data structure for holding RGB color values
//! \ingroup light
/*! Each LRFColorValue is comprised of three bytes of data, one for each red, green and blue. The maximum value (full brightness) for each byte is 63, however we are typically using 40 to represent a high-brightness value (to save battery power). Low brightness is current defined as a value of 5. To create a custom color, play around with the values defined in LRFUtils.cpp until you've got a good match, then you can rename a LRFColor constant to use your custom color in your code. Eventually we will have a method for loading custom colors from outside this library, but not at the moment. */
typedef struct
{
	unsigned char red;		//!< Red value (8-bits)
	unsigned char green;	//!< Green value (8-bits)
	unsigned char blue;		//!< Blue value (8-bits)
} __attribute__ ((packed)) LRFColorValue;

//! \enum LRFPatternMode
//! \brief Predefined pattern animations
//! \ingroup light
/*! Every pattern contains a pattern 'mode' that defines how it animates between the start and target colors (LRFColor). There are a handful of animations that cover a wide range of possibilties. 
	- Flat patterns (eg: LRFPatternMode_Flat) will simply load the starting color, then do nothing.
	- Fade patterns (eg: LRFPatternMode_Fade) will load the starting color into both eyes, then fade both eyes to the target color. 
	- Toggle patterns (eg: LRFPatternMode_FadeToggle) will load the starting color into the left eye and the target color into the right eye, then perform its animation. 
	- Flip patterns (eg: LRFPatternMode_FadeFlip) will load the starting color, animate to the target color, and then flip 'start' and 'target' when the pattern is reset. This is handy when expressing a set of sounds and can add some complexity to your custom expression. 
	- Boomerang patterns (eg: LRFPatternMode_Boomerang) fade from the starting color to the target color, then back again.
	- Random patterns (eg: LRFPatternMode_RandomFlat) choose random start and target colors for a fun surprise.
 
	Check out the list of patterns below for a full explaination, and of course - these patterns can always be customized further!
 */
typedef enum
{
	LRFPatternMode_Flat = 0,		//!< Only use starting color with no change
	LRFPatternMode_FlatFlip,		//!< Use starting colour, flip start and target on reset pattern
	LRFPatternMode_FlatFlipToggle,	//!< Left=starting color, Right=target color, flip on reset
	LRFPatternMode_Fade,			//!< Fade from start to target
	LRFPatternMode_FadeFlip,		//!< Fade from start to target, flip on reset
	LRFPatternMode_FadeToggle,		//!< Left=starting, Right=target, fade
	LRFPatternMode_FadeFlipToggle,	//!< Left=starting, Right=target, fade and flip on reset
	LRFPatternMode_Boomerang,		//!< Fade from start to target then back to start
	LRFPatternMode_BoomerangToggle,	//!< Left=starting, right=target, fade to opposite and back
	LRFPatternMode_RandomFlat,		//!< Random start and target colours with flat effect
	LRFPatternMode_RandomFade,		//!< Random start and target colours with fade effect
	LRFPatternMode_RandomToggle,	//!< Random start and target colours with toggle effect
	LRFPatternMode_RandomBoomerang,	//!< Random start and target colours with boomerang effect
	LRFPatternMode_None = 15		//!< No pattern selected
} __attribute__ ((packed)) LRFPatternMode;

//! \struct	LRFPatternStruct
//! \brief	Structure for driving LED light patterns
//! \ingroup light
/*! This tightly-packed data structure contains all the information required for animating colors on the LRFs RGB LED eyes. By using lookup tables for color and animation data, the entire structure takes up only 12-bits (with 4-bits reserved for future use). That means you can create a lot of different patterns without taking up a ton of precious memory. The start and target members (LRFColor) define the 'a' and 'b' values for the patterns animation (as defined by 'mode'). Creating a custom pattern looks like this:
 
		LRFPatternStruct myPattern = { LRFColor_Red, LRFColor_Green, LRFPatternMode_Fade };
 
	This generates a pattern where both eyes will begin as red, then fade to green over the patterns duration (which defaults to 1000ms). 
 
	A more complex pattern might be:
 
		LRFPatternStruct myOtherPattern = { LRFColor_Red, LRFColor_Green, LRFPatternMode_FadeFlipToggle };
 
	In this case, the left eye begins as red, the right eye begins as green, then they crossfade colors. Once the pattern completes and is reset, the start and target colors 'flip'. On the next run of that pattern the left eye will begin as green and the right eye will begin as red. And so on, and so on...
 
	You can also set and get the values of the structure separately...
 
		LRFColor color = myPattern.start;	// copy the starting color into a local variable
		myPattern.start = LRFColor_Blue;	// change the starting color to blue
 
 */
typedef struct
{
	LRFColor				start:4;		//!< Starting color
	LRFColor				target:4;		//!< Target color
	LRFPatternMode			mode:4;			//!< Animation type (perhaps 'type' would have been a better name)
	int						reserved:4;		//!< Reserved for future use (or if you need some extra bits?)
} __attribute__ ((packed)) LRFPatternStruct;

//! \union	LRFPattern
//! \brief	Allows for access of pattern data as a structure or raw integer
//! \ingroup light
/*! Pattern data structures are wrapped in this union datatype to allow for easy access to the raw data contained therein. Patterns can be created using the same method as creating a LRFPatternStruct. For example:
 
		LRFPattern myPattern = { LRFColor_Red, LRFColor_Green, LRFPatternMode_Fade };
 
	Is the same as...
 
		LRFPatternStruct myPattern = { LRFColor_Red, LRFColor_Green, LRFPatternMode_Fade };
 
	However, with our union type I can also access the raw integer value of that pattern.
 
		Serial.print(myPattern.raw,BIN);				// prints the integer value in binary format
		myPattern.raw = pgm_read_word(&mySavedPattern);	// loads pattern data from program memory
	
	For more information on how patterns work, check out the documentation for LRFPatternStruct.
 
 */
typedef union
{
	LRFPatternStruct pattern;
	int raw;
} __attribute__ ((packed)) LRFPattern;


//-------------------------------------------------------------
#pragma mark - Sound Types
//-------------------------------------------------------------

//! \enum	LRFNote
//! \brief	Constants for notes in the musical scale
//! \ingroup sound
/*! Most sounds uttered by the LRF are pinned to the frequencies of common musical notes. These frequencies are stored in program memory (in LRFUtils.cpp) as a lookup table. 

	This enum is packed to 4-bits (with 13 possible values)
 */
typedef enum
{
	LRFNote_A	= 0,	//!< A	(55 hertz at octave #1)
	LRFNote_AS,			//!< A#	(58hz)
	LRFNote_B,			//!< B	(62hz)
	LRFNote_C,			//!< C	(65hz)
	LRFNote_CS,			//!< C#	(69hz)
	LRFNote_D,			//!< D	(73hz)
	LRFNote_DS,			//!< D#	(78hz)
	LRFNote_E,			//!< E	(82hz)
	LRFNote_F,			//!< F	(87hz)
	LRFNote_FS,			//!< F#	(92hz)
	LRFNote_G,			//!< G	(98hz)
	LRFNote_GS,			//!< G#	(104hz)
	LRFNote_NULL = 15	//!< Catching undefined notes
} __attribute__ ((packed)) LRFNote;

//! \enum	LRFOctave
//! \brief	Constants for octaves in the musical scale
//! \ingroup sound
/*! The octave constants allow for shifting our stored note frequencies into higher ranges for a wider variety of sounds. Not all notes are playable for all octaves, since some of them overload the frequency timer - so it's kinda trial and error. 
 
	This enum is packed to 3-bits (with 7 possible values)
*/
typedef enum
{
	LRFOctave_1	= 0,	//!< lowest possible octave (some notes are unplayable)
	LRFOctave_2,		//!< lower octave
	LRFOctave_3,		//!< middle-low octave
	LRFOctave_4,		//!< middle octave
	LRFOctave_5,		//!< middle-high octave
	LRFOctave_6,		//!< high octave
	LRFOctave_7,		//!< higher octave
	LRFOctave_8			//!< highest possible octave
} __attribute__ ((packed)) LRFOctave;

//! \enum	LRFIntonation
//! \brief	Constants for speech intonation (or pitch bend)
//! \ingroup sound
/*! In order to make LRF sounds have extra character, the frequency of each sound can be altered over the duration of its playback. This is akin to intonation in speech. By combining a variety of notes, intonations and durations you can get some really complex expressions!
 
	This enum is packed to 3-bits (with 5 possible values)
 */
typedef enum
{
	LRFIntonation_Flat = 0,		//!< Flat tone (no change)
	LRFIntonation_Rising = 1,	//!< Rising tone (happy or excited)
	LRFIntonation_Falling = 2,	//!< Falling tone (sad or upset)
	LRFIntonation_Peaking = 3,	//!< Peaking tone (tone rises, then returns to normal)
	LRFIntonation_Dipping = 4	//!< Dipping tone (tone dips, then returns to normal)
} __attribute__ ((packed)) LRFIntonation;

/*! \enum	LRFDuration
	\brief	Constants for fixed note durations based on musical timing
	\ingroup sound
 
	We found that a short list of fixed durations covered a pretty wide range of sound timings that would be required when created new sounds. By taking this approach, we are also able to pack a lot more sound data into a smaller space - meaning more sounds for all! These constants map to a lookup table (in LRFUtils.h) that contains the actual duration values in milliseconds (as an unsigned int).
 
	This enum is packed to 3-bits (with 8 possible values)
 */
typedef enum
{
	LRFDuration_None = 0,		//!< No duration (0 ms)
	LRFDuration_Short,			//!< Base * 1 (1/16 note)
	LRFDuration_DoubleShort,	//!< Base * 2 (1/8 note)
	LRFDuration_Medium,			//!< Base * 4 (1/4 note)
	LRFDuration_DoubleMedium,	//!< Base * 8 (1/2 note)
	LRFDuration_Long,			//!< Base * 16 (whole note)
	LRFDuration_DoubleLong,		//!< Base * 32 (double whole note)
	LRFDuration_VeryLong		//!< Base * 64 (quad whole note)
} __attribute__ ((packed)) LRFDuration;

/*! \struct	LRFSoundStruct
	\brief	Structure for driving expressive robot sounds
	\ingroup sound
 
	\details This tightly-packed data structure contains all the information require to speak a single sound instance. We approached the LRF capability for sound as if we were designing a library for robot speech. Each data structure contains:

	- 'note' and 'octave' values for selecting a frequency (tone)
	- an 'intonation' value for bending the frequency during playback
	- 'duration' and 'pause' values for controlling how long the note plays and how long to pause after it finishes playing
 
	By combining these sounds into sets (using arrays) you can make your robot express in an almost infinite number of ways. 
 
	Sounds can be created in a single line of code:

		LRFSoundStruct mySound = { LRFNote_C, LRFOctave_4, LRFIntonation_Rising, LRFDuration_Medium, LRFDuration_None };
 
	Or using multiple lines of code:
 
		LRFSoundStruct mySound;
		mySound.note = LRFNote_C;
		mySound.octave = LRFOctave_4;
		mySound.intonation = LRFIntonation_Rising;
		mySound.duration = LRFDuration_Medium;
		mySound.pause = LRFDuration_None;
 
	You can play around with the various values to find sounds that are happy, sad, angry, excited... almost anything! Since the data packs tightly into a 16-bit integer (with 1 bit to spare), we can store a lot of custom sounds for your Little Robot Friend.
 */
typedef struct
{
	LRFNote			note:4;			//!< musical note (base frequency)
	LRFOctave		octave:3;		//!< musical octave (scale the frequency)
	LRFIntonation	intonation:3;	//!< bend applied to the frequency (over duration_
	LRFDuration		duration:3;		//!< time the note is played
	LRFDuration		pause:3;		//!< time that we rest after the note is played
} __attribute__ ((packed)) LRFSoundStruct;

/*! \def LRFSoundStruct_Blank
	\brief an empty sound (used frequenty in LRFExpressionData and LRFSignatureData)
	\ingroup sound
 */
#define LRFSoundStruct_Blank		{LRFNote_NULL,0,0,0,0}

/*! \union	LRFSound
	\brief	Allows for access of sound data as a structure or raw integer
	\ingroup sound
 
	\details Sound data structures are wrapped in this union datatype to allow for easy access to the raw data contained therein. Sounds can be created using the same method as creating a LRFSoundStruct. For example:
 
		LRFSound mySound = { LRFNote_C, LRFOctave_4, LRFIntonation_Rising, LRFDuration_Medium, LRFDuration_None };
 
	Is the same as...
 
		LRFSoundStruct mySound = { LRFNote_C, LRFOctave_4, LRFIntonation_Rising, LRFDuration_Medium, LRFDuration_None };
 
	However, with our union type we can also access the raw integer value of that sound.
 
		Serial.print(mySound.raw,BIN);				// prints the integer value in binary format
		mySound.raw = pgm_read_word(&mySavedSound);	// loads sound data from program memory
 
	For more information on how sounds work, check out the documentation for LRFSoundStruct.
 */
typedef union
{
	LRFSoundStruct sound;
	int raw;
} __attribute__ ((packed)) LRFSound;

// LRFFunctionPtr - General function pointer object
typedef void (*LRFFunctionPtr)(void);


//-------------------------------------------------------------
#pragma mark - Lookup Functions
//-------------------------------------------------------------

/*! \fn		lrf_utils_color_lookup(LRFColor color, LRFColorValue *value)
	\brief	Utility for looking up LRFColorValue data from LRFColor constant
	\ingroup light
	\param	color The LRFColor constant to lookup
	\param	value A pointer to a LRFColorValue variable where the value will be written
 
	Usage:
		
		LRFColorValue color;		// define the variable for holding the value
		lrf_utils_color_lookup(LRFColor_Red, &color);	// pass the red constant AND the address to our 'color' variable
		Serial.print(color.red,DEC)	// print the red parameter for our color (should be 40)
 */
void lrf_utils_color_lookup(LRFColor color, LRFColorValue *value);

/*! \fn		lrf_utils_color_lookup_random(LRFColorValue *value)
	\brief	Utility for generating a random LRFColorValue
	\ingroup light
	\param	value A pointer to a LRFColorValue variable where the value will be written

	Usage:
 
		LRFColorValue color;		// define the variable for holding the value
		lrf_utils_color_lookup_random(&color);	// pass the address to our 'color' variable
		Serial.print(color.red,DEC)	// print the red parameter for our color (should be ???)
 */
void lrf_utils_color_lookup_random(LRFColorValue *value);

/*! \fn		unsigned int lrf_utils_duration_lookup(LRFDuration duration)
	\brief	Utility for looking up duration values (in milliseconds) from a LRFDuration constant
	\ingroup sound
	\param	duration The LRFDuration constant to lookup
	\return	Duration value in milliseconds
 
	Usage:
 
		unsigned int duration;		// define the variable for holding the value
		duration = lrf_utils_duration_lookup(LRFDuration_Medium);	// lookup the medium duration value from program memory
		Serial.print(duration,DEC)	// print the medium duration in milliseconds (should be 200)
 */
unsigned int lrf_utils_duration_lookup(LRFDuration duration);

/*! \fn		unsigned int lrf_utils_note_to_frequency(LRFNote note, LRFOctave octave)
	\brief	Utility for looking up frequency values from LRFNote and LRFOctave constants
	\ingroup sound
	\param	note The musical note constant
	\param	octave The musical octave constant
	\return	Timer value for generating a tone at a specified frequency

	Usage:

		unsigned int freq;		// define the variable for holding the value
		freq = lrf_utils_note_to_frequency(LRFNote_C, LRFOctave_3);	// lookup the frequency value from program memory
		Serial.print(freq,DEC)	// print the value
 
		lrf_speaker_on();					// turn on the speaker
		lrf_speaker_set_frequency(freq);	// set the frequency
		delay(1000);						// delay for 1s
		lrf_speaker_off();					// turn off the speaker
 */
unsigned int lrf_utils_note_to_frequency(LRFNote note, LRFOctave octave);

bool lrf_utils_is_holiday_hardware(void);


#endif