//
//  LRFPower.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 2014-03-14.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#include "LRFPower.h"

#include <Arduino.h>
#include <avr/sleep.h>
#include <avr/power.h>
#include "LRFEvents.h"
#include "LRFHardware.h"
#include "LRFMemory.h"
#include "LRFDebug.h"
#include "LRFSensors.h"
#include "LRFEyes.h"


const char debug_title[] = "pow";

volatile LRFPower lrfPower = { false, false, 0 };


//-------------------------------------------------------------
#pragma mark - Setup & Process
//-------------------------------------------------------------

void lrf_power_setup(void)
{
	set_sleep_mode(SLEEP_MODE_PWR_DOWN);
	power_twi_disable();
	power_spi_disable();
	
	// Reduce power on AC pins
	DIDR1 = 0b00000011;
	
	// (Disable) ACME: Analog Comparator Multiplexer Enable
	ADCSRB = 0;

	// (Clear) Analog Comparator Interrupt Flag
	// Analog Comparator Interrupt Enable
	// ACIS1, ACIS0: Analog Comparator Interrupt Mode Select (trigger on falling edge)
//	ACSR =  (1 << ACI) | (1 << ACIE) | (1 << ACIS1);

	// clear comparator interrupt flag, disable comparator
	ACSR = (1<<ACD) | (1<<ACI);
	
	lrfPower.nextFuelCheck = millis() + 5000;
	lrfPower.isHungry = !lrf_power_fuel_check();
}

bool lrf_power_fuel_check(void)
{
	bool result = true;
	
#if LRF_POWER_DEBUG
	lrf_debug_tag(debug_title, "fuel:", false);
#endif
	
	// enable the comparator
	ACSR &= ~(1<<ACD);
	
	// short delay
	delayMicroseconds(5);
	
	// read the comparator
	if(ACSR & (1<<ACO))
	{
#if LRF_POWER_DEBUG
		Serial.println("ok");
#endif
		result = true;
	}
	else
	{
#if LRF_POWER_DEBUG
		Serial.println("low");
#endif
		result = false;
	}
	
	// disable the comparator
	ACSR |= (1<<ACD);
	
	return result;
}

void lrf_power_process(void)
{
	if(millis() > lrfPower.nextFuelCheck)
	{
		lrfPower.nextFuelCheck = millis() + 10000; // check power every 10s
		
		bool isHungry = !lrf_power_fuel_check();
				
		if(isHungry)
		{
			// we're outta fuel!
			lrf_events_add(LRFEvent_IsHungry);
		}
		else
		{
			// phew!
			if(lrfPower.isHungry) lrf_events_add(LRFEvent_PowerUp);
		}
		
		lrfPower.isHungry = isHungry;
	}
}


//-------------------------------------------------------------
#pragma mark - Sleep & Wake
//-------------------------------------------------------------

// ** sleep needs to make sure LEDs / Speakers aren't drawing power
// ** also needs to detatch timer2 interrupt driving leds and replace with wake-up timer

void lrf_power_sleep(void)
{
	// just in case we're going back to sleep from false wake
	if(lrfPower.isSleeping == false)
	{
		// turn off leds
		lrf_rgb_disable();
		
		// turn off sensors
		lrf_sensors_disable();
		
		// save our states
		lrf_memory_save();
	}
		
	// set our sleeping flag
	lrfPower.isSleeping = true;
	
#if LRF_POWER_DEBUG
	lrf_debug_tag(debug_title, "sleep:", false);
	Serial.println(millis(),DEC);
#endif
	delay(2000);

	// disable all non-essential peripherals
	power_adc_disable();	// sensors
	power_usart0_disable();	// serial
	power_timer1_disable();	// speaker timer

	// disable bods
	MCUCR |= (1 << BODS) | (1 << BODSE);
	MCUCR &= ~(1 << BODSE);
	
	while(lrfPower.isSleeping)
	{
		lrf_eyes_set_color(LRFColor_Blank);
		lrf_rgb_disable();
		
		// enable INT0 for waking
		attachInterrupt(0, lrf_power_wake, RISING);
		
		// disable power for a bunch of things
		power_timer0_disable();	// delay timer
		power_timer2_disable(); // leds timer
		
		// sleep the cpu
		sleep_enable();
		sleep_cpu();
		
		// SLEEPING....
		
		power_timer0_enable();
		power_timer2_enable();
		
		delay(100);

		// disable our interrupt
		detachInterrupt(0);

		unsigned char count = 50;
		unsigned char val = 0;
		lrf_touch_read(&val);

		// extra check for holiday hardware
		if(lrf_utils_is_holiday_hardware())
		{
			// step through timer for when we should turn on
			while(count > 0 && val)
			{
				lrf_touch_read(&val);
				delay(50);
				count--;
			}
			
			if(count > 0) continue;
			
			// extra delay for holiday
			delay(500);
			count = 120;
		}
		
		lrf_rgb_enable();
		lrf_eyes_set_color(LRFColor_SoftBlue);
		
		while(count > 0 && val)
		{
			lrf_touch_read(&val);
			delay(50);
			count--;
		}
		
#if LRF_POWER_DEBUG
		power_usart0_enable();
		Serial.println(count,DEC);
		delay(500);
		power_usart0_disable();
#endif
	
		if(count == 0) lrfPower.isSleeping = false;
	}
	
	// re-enable peripherals
	power_adc_enable();
	power_usart0_enable();
	power_timer1_enable();

	lrf_rgb_enable();
	lrf_events_update_interaction();
	lrf_sensors_enable();
	lrf_sensors_recalibrate();
	
#if LRF_POWER_DEBUG
	lrf_debug_tag(debug_title, "wake:", false);
	Serial.println(millis(),DEC);
	delay(100);
#endif
	
	lrf_events_add(LRFEvent_Wake);
}

void lrf_power_wake(void)
{
	sleep_disable();
}

