//
//  LRFHardware.h
//  LittleRobotFriends
//
//  Created by Mark Argo on 1/14/2014.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#ifndef __LRFHardware_h
#define __LRFHardware_h

#include <Arduino.h>

#define TEST_RGB			0
#define TEST_SPEAKER		0
#define TEST_TOUCH			0

//------------------------------------------------------------
#pragma mark - Hardware Defines
//------------------------------------------------------------

// B0: Left Green LED	(output, active high)
// B1: Speaker Out		(output, OC1A PWM'd)
// B2: Right Red LED	(output, active high)
// B3: Left Blue LED	(output, active high)
// B4: Right Green LED	(output, active high)
// B5: Right Blue LED	(output, active high)
// B6: Not Connected
// B7: Left Red LED		(output, active high)
#define LRF_B_DIR			0b10111111		// all output (except B6)
#define LRF_B_PORT			0b00000000		// all low on init

// C0: Shoelace 0		(input, ADC0)
// C1: Shoelace 1		(input, ADC1)
// C2: Shoelace 2		(input, ADC2)
// C3: Shoelace 3		(input, ADC3)
// C4: Shoelace 4		(input, ADC4)
// C5: Shoelace 5		(input, ADC5)
// C6: Microphone In	(input, ADC6)
// C7: Light Sensor In	(input, ADC7)
#define LRF_C_DIR			0b00000000		// all input
#define LRF_C_PORT			0b00000000		// any pullups?

// D0: Serial In		(input, RX)
// D1: Serial Out		(output, TX)
// D2: Touch In			(input, INT0)
// D3: IrDA In			(input, INT1, pullup)
// D4: IrDA Out			(output, active low)
// D5: Touch Sync		(output, active low)
// D6: Battery +		(input, AIN0)
// D7: Battery Test		(input, AIN1)
#define LRF_D_DIR			0b00110010
#define LRF_D_PORT			0b00011000

#define LRF_MIC_DIR			DDRC
#define LRF_MIC_PIN			6
#define LRF_MIC_ADC			6

#define LRF_IRDA_PORT		PORTD
#define LRF_IRDA_OUT		0b00010000
#define LRF_IRDA_IN			0b00001000


// LED Masks
#define LRF_RGB_PORT			PORTB
#define LRF_RGB_LEFT_BLUE		0b00000001
#define LRF_RGB_LEFT_GREEN		0b00001000
#define LRF_RGB_LEFT_RED		0b10000000
#define LRF_RGB_RIGHT_BLUE		0b00010000
#define LRF_RGB_RIGHT_GREEN		0b00100000
#define LRF_RGB_RIGHT_RED		0b00000100

#define LRF_TOUCH_IN		0b00000100
#define LRF_TOUCH_SYNC		0b00100000

#define LRF_RGB_VALUE_MAX	64
#define LRF_RGB_VALUE_HIGH	40
#define LRF_RGB_VALUE_MED	20
#define LRF_RGB_VALUE_SOFT	5

//------------------------------------------------------------
#pragma mark - Type Definitions
//------------------------------------------------------------

typedef enum
{
	LRFRgbMask_LeftRed =	0b00000100,
	LRFRgbMask_LeftGreen =	0b00100000,
	LRFRgbMask_LeftBlue =	0b00010000,
	LRFRgbMask_RightRed =	0b10000000,
	LRFRgbMask_RightGreen =	0b00001000,
	LRFRgbMask_RightBlue =	0b00000001,
	LRFRgbMask_LeftAll =	0b00110100,
	LRFRgbMask_RightAll =	0b10001001,
	LRFRgbMask_All =		0b10111101
} __attribute__ ((packed)) LRFRgbMask;


//------------------------------------------------------------
#pragma mark - General Functions
//------------------------------------------------------------

void lrf_hardware_setup(void);

void lrf_hardware_test(bool keyboard=false);
void lrf_hardware_advanced_test(void);
void lrf_hardware_body_test(void);


//------------------------------------------------------------
#pragma mark - Speaker Functions
//------------------------------------------------------------

void lrf_speaker_setup(void);
void lrf_speaker_on(void);
void lrf_speaker_off(void);
inline void lrf_speaker_set_frequency(unsigned int freq)
{
	OCR1AH = freq >> 8;
	OCR1AL = freq;
}


//------------------------------------------------------------
#pragma mark - RGB Functions
//------------------------------------------------------------

void lrf_rgb_setup(void);
void lrf_rgb_set(LRFRgbMask mask, unsigned char duty);
void lrf_rgb_process(void);
void lrf_rgb_enable(void);
void lrf_rgb_disable(void);


//------------------------------------------------------------
#pragma mark - IrDA
//------------------------------------------------------------

inline void lrf_irda_write_low()
{
	LRF_IRDA_PORT |= LRF_IRDA_OUT;
}

inline void lrf_irda_write_high()
{
	LRF_IRDA_PORT &= ~LRF_IRDA_OUT;
}

inline void lrf_irda_read_enable()
{
	EIFR = 0b00000010;
	EIMSK |= 0b00000010;
}

inline void lrf_irda_read_disable()
{
	EIMSK &= 0b11111101;
}

//------------------------------------------------------------
#pragma mark - Input Reading Functions
//------------------------------------------------------------

void lrf_irda_read(unsigned char *value);
void lrf_touch_read(unsigned char *value);
void lrf_mic_read(unsigned char *value);
void lrf_light_read(unsigned char *value);


//------------------------------------------------------------
#pragma mark - Test Functions
//------------------------------------------------------------

#if TEST_SPEAKER
void lrf_speaker_test(void);
#endif

#if TEST_RGB
void lrf_rgb_test(void);
#endif

#if TEST_TOUCH
void lrf_touch_test(void);
#endif

#endif
