//
//  LRFHardware.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 1/14/2014.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#include "LRFHardware.h"
#include "LRFMemory.h"
#include "LRFInfrared.h"
#include "LRFUtils.h"
#include "LRFEyes.h"

//------------------------------------------------------------
#pragma mark - General Functions
//------------------------------------------------------------

void lrf_hardware_setup(void)
{
	// setup the hardware
	DDRB = LRF_B_DIR;
	DDRC = LRF_C_DIR;
	DDRD = LRF_D_DIR;
	PORTB = LRF_B_PORT;
	PORTC = LRF_C_PORT;
	PORTD = LRF_D_PORT;
	
	lrf_rgb_setup();
	lrf_speaker_setup();
}


//------------------------------------------------------------
#pragma mark - Speaker
//------------------------------------------------------------

void lrf_speaker_on(void)
{
	TCCR1A = 0b01000011;	// OC1A toggle OC1B normal, fast pwm 8-bit, OCR1A top, f/1
	TCCR1B = 0b00011001;
	OCR1AH = 0;
	OCR1AL = 0;
}

void lrf_speaker_off(void)
{
	TCCR1A = 0b00000000;	// OC1A toggle OC1B normal, fast pwm 8-bit, OCR1A top, f/1
	TCCR1B = 0b00000000;
	PORTB &= ~0b00000010;
}

void lrf_speaker_setup(void)
{
	lrf_speaker_off();
}


//------------------------------------------------------------
#pragma mark - RGB
//------------------------------------------------------------

unsigned char lrf_rgb_duty_map[LRF_RGB_VALUE_MAX];
unsigned char lrf_rgb_duty_idx=0;

void lrf_rgb_set(LRFRgbMask mask, unsigned char duty)
{
	unsigned char i=0;
	if(duty > LRF_RGB_VALUE_MAX) duty = LRF_RGB_VALUE_MAX;
	while(i<duty)
	{
		lrf_rgb_duty_map[i++] |= mask;		// turn on
	}
	while(i<LRF_RGB_VALUE_MAX)
	{
		lrf_rgb_duty_map[i++] &= ~mask;		// turn off
	}
}

void lrf_rgb_enable(void)
{
	TIMSK2 = 0b00000001;
}

void lrf_rgb_disable(void)
{
	lrf_rgb_set(LRFRgbMask_All,0);
	TIMSK2 = 0b00000000;
}

void lrf_rgb_setup(void)
{
	lrf_rgb_set(LRFRgbMask_All,0);
	
	TCCR2A = 0b00000011;
	TCCR2B = 0b00001001;
	OCR2A = 0x80;
	
	lrf_rgb_enable();
}

void lrf_rgb_process(void)
{
	LRF_RGB_PORT = lrf_rgb_duty_map[lrf_rgb_duty_idx];
	lrf_rgb_duty_idx = (lrf_rgb_duty_idx+1) % LRF_RGB_VALUE_MAX;
}

ISR(TIMER2_OVF_vect)
{
	lrf_rgb_process();
}

//------------------------------------------------------------
#pragma mark - Input Reading Functions
//------------------------------------------------------------

void lrf_irda_read(unsigned char *value)
{
	*value = (PIND & LRF_IRDA_IN) == 0;
}

void lrf_touch_read(unsigned char *value)
{
	*value = (PIND & LRF_TOUCH_IN) > 0;
}

void lrf_mic_read(unsigned char *value)
{
	unsigned char foo;
	
	// switch pin back to input
	bitClear(LRF_MIC_DIR, LRF_MIC_PIN);
		
	ADMUX = LRF_MIC_ADC;				// set the mic channel
	ADCSRA |= (1<<ADSC) | (1<<ADEN);	// trigger an ADC read
	while(bit_is_set(ADCSRA, ADSC));	// wait for read to complete
	
	*value = ADCL;
	foo = ADCH;							// need to trash high bits
	
//	ADMUX = 0b00001111;					// set the channel to gnd
//	ADCSRA |= (1<<ADSC) | (1<<ADEN);	// trigger an ADC read
//	while(bit_is_set(ADCSRA, ADSC));	// wait for read to complete
//	foo = ADCH;

	// switch pin to output to drain
//	bitSet(LRF_MIC_DIR, LRF_MIC_PIN);
}

void lrf_light_read(unsigned char *value)
{
	unsigned int sample;
	
	ADMUX = 0b00000111;					// set the mic channel
	ADCSRA |= (1<<ADSC) | (1<<ADEN);	// trigger an ADC read
	while(bit_is_set(ADCSRA, ADSC));	// wait for read to complete
	
	// switch ADC alignment HERE
	
	sample = ADCL;
	sample = (ADCH << 8) | sample;
	
	*value = (sample >> 2) & 0xff;
}



//------------------------------------------------------------
#pragma mark - Test Functions
//------------------------------------------------------------

void lrf_hardware_body_test()
{
	unsigned char touch = 0;
	unsigned char color = 0;
	unsigned int freq = 0;
	
	while(true)
	{
		Serial.println("test");
		
		while(touch == 0)
		{
			lrf_touch_read(&touch);
			delay(10);
		}
		
		Serial.println("on");
		
		while(touch == 1)
		{
			lrf_touch_read(&touch);
			delay(10);
		}
		
		Serial.println("off");

		if(color == 0)		lrf_eyes_set_color(LRFColor_Red);
		else if(color == 1)	lrf_eyes_set_color(LRFColor_Green);
		else if(color == 2)	lrf_eyes_set_color(LRFColor_Blue);
		
		freq = 6000 - (color*1000);
		lrf_speaker_on();
		lrf_speaker_set_frequency(freq);
		delay(500);
		lrf_speaker_off();
		
		color = (color+1) % 3;
	}
}

void lrf_hardware_test_advanced(void)
{
	unsigned char tempA = 0;
	unsigned char tempB = 0;
	unsigned int tempInt = 0;
	bool passed = false;
	
	Serial.println("LRF Hardware Test");
	delay(1000);
	
	lrf_irda_setup();
	lrf_rgb_disable();
	//	lrf_irda_read_disable();
	//	lrf_irda_read_enable();
	interrupts();
	
	while(true)
	{
		// ------------------------------------------- Touch
		Serial.print("touch sensor...");
		
		
		// can touch go on?
		passed = false;
		while(!passed)
		{
			lrf_touch_read(&tempB);
			Serial.print(tempB,DEC);
			delay(100);
			if(tempB != 0) passed = true;
			
			if(lrfIrda.hasMessage)
			{
				Serial.print("[msg:");
				Serial.print(lrfIrda.message,DEC);
				Serial.print("]");
				lrfIrda.hasMessage = false;
				lrf_irda_read_enable();
			}
		}
		if(passed)
		{
			Serial.println(" passed on.");
			LRF_RGB_PORT |= LRF_RGB_LEFT_RED;
		}
		
		// can touch go off?
		passed = false;
		while(!passed)
		{
			lrf_touch_read(&tempB);
			Serial.print(tempB,DEC);
			delay(100);
			if(tempB == 0) passed = true;
			
			if(lrfIrda.hasMessage)
			{
				Serial.print("[msg:");
				Serial.print(lrfIrda.message,DEC);
				Serial.print("]");
				lrfIrda.hasMessage = false;
				lrf_irda_read_enable();
			}
		}
		if(passed)
		{
			Serial.println(" passed off.");
			LRF_RGB_PORT = 0;
		}
	}
	
	while(true) delay(10);
}

void lrf_hardware_test(bool keyboard)
{
	unsigned char tempA = 0;
	unsigned char tempB = 0;
	unsigned int tempInt = 0;
	bool passed = false;
	bool done = false;
	
	Serial.println("LRF Hardware Test");
	delay(1000);

	while(!done)
	{
		// ------------------------------------------- Touch
		Serial.print("touch sensor...");
		
		lrf_rgb_disable();
		
		// can touch go on?
		passed = false;
		while(!passed)
		{
			lrf_touch_read(&tempB);
			Serial.print(tempB,DEC);

			LRF_RGB_PORT |= LRF_RGB_LEFT_RED;
			delay(5);
			LRF_RGB_PORT = 0;
			delay(500);

			if(keyboard){ if(Serial.available()) { Serial.read(); passed = true; } }
			else		{ if(tempB != 0) passed = true; }
		}
		if(passed)
		{
			Serial.println(" passed on.");
			LRF_RGB_PORT |= LRF_RGB_LEFT_RED;
		}
		
		// can touch go off?
		passed = false;
		while(!passed)
		{
			lrf_touch_read(&tempB);
			Serial.print(tempB,DEC);

			LRF_RGB_PORT |= LRF_RGB_RIGHT_RED;
			delay(5);
			LRF_RGB_PORT = 0;
			delay(500);

			if(keyboard){ if(Serial.available()) { Serial.read(); passed = true; } }
			else		{ if(tempB == 0) passed = true; }
		}
		if(passed)
		{
			Serial.println(" passed off.");
			LRF_RGB_PORT = 0;
		}
		
		// ------------------------------------------- Light
		Serial.print("light sensor...");
		
		// low test
		passed = false;
		while(!passed)
		{
			lrf_light_read(&tempA);
			Serial.print(tempA,DEC);
			Serial.print(",");
			delay(500);
			if(keyboard){ if(Serial.available()) { Serial.read(); passed = true; } }
			else		{ if(tempA < 20) passed = true; }
		}
		if(passed) Serial.println(" passed low.");
		
		// high test
		passed = false;
		while(!passed)
		{
			lrf_light_read(&tempA);
			Serial.print(tempA,DEC);
			Serial.print(",");
			delay(500);
			if(keyboard){ if(Serial.available()) { Serial.read(); passed = true; } }
			else		{ if(tempA > 230) passed = true; }
		}
		if(passed) Serial.println(" passed high.");
		
		// ------------------------------------------- Microphone
		Serial.print("microphone...");
		passed = false;
		lrf_mic_read(&tempB);
		while(!passed)
		{
			lrf_mic_read(&tempA);
			tempInt = abs(tempB-tempA);
			
//			Serial.print(tempA,DEC);
//			Serial.print(",");
//			Serial.print(tempB,DEC);
//			Serial.print(",");
			Serial.print(tempInt,DEC);
			Serial.print(",");
			delay(100);
			
			if(keyboard){ if(Serial.available()) { Serial.read(); passed = true; } }
			else		{ if(tempInt > 20) passed = true; }
//			else		{ if(tempA < 5 || tempA > 80) passed = true; }
		}
		if(passed) Serial.println(" passed.");
		
		// ------------------------------------------- Irda
		Serial.print("irda...");
		passed = false;
		while(!passed)
		{
			tempB = 0;
			
			while(tempB != 0b10101010)
			{
				// put the irda high
				lrf_irda_write_high();
				delay(100);
				lrf_irda_read(&tempA);
				Serial.print(tempA,DEC);
				Serial.print(",");
				delay(200);
				tempB = (tempB << 1) | tempA;
				
				// put the irda low
				lrf_irda_write_low();
				delay(100);
				lrf_irda_read(&tempA);
				Serial.print(tempA,DEC);
				Serial.print(",");
				delay(200);
				tempB = (tempB << 1) | tempA;
			}
			
			if(keyboard){ if(Serial.available()) { Serial.read(); passed = true; } }
			else		{ passed = true; }
			
		}
		if(passed) Serial.println(" passed.");
		
		// ------------------------------------------- Leds
		Serial.print("leds...");
		passed = false;
		lrf_rgb_disable();
		while(!passed)
		{
			LRF_RGB_PORT |= LRF_RGB_LEFT_RED;
			delay(500);
			LRF_RGB_PORT = 0;
			delay(100);
			
			LRF_RGB_PORT |= LRF_RGB_LEFT_GREEN;
			delay(500);
			LRF_RGB_PORT = 0;
			delay(100);
			
			LRF_RGB_PORT |= LRF_RGB_LEFT_BLUE;
			delay(500);
			LRF_RGB_PORT = 0;
			delay(100);
			
			LRF_RGB_PORT |= LRF_RGB_RIGHT_RED;
			delay(500);
			LRF_RGB_PORT = 0;
			delay(100);
			
			LRF_RGB_PORT |= LRF_RGB_RIGHT_GREEN;
			delay(500);
			LRF_RGB_PORT = 0;
			delay(100);
			
			LRF_RGB_PORT |= LRF_RGB_RIGHT_BLUE;
			delay(500);
			LRF_RGB_PORT = 0;
			delay(100);
			
			lrf_touch_read(&tempB);
			
			if(keyboard){ if(Serial.available()) { Serial.read(); passed = true; } }
			else		{ if(tempB != 0) passed = true; }
		}
		lrf_rgb_enable();
		if(passed) Serial.println(" passed.");
		
		// ------------------------------------------- Speaker
		Serial.print("speaker...");
		passed = false;
		while(!passed)
		{
			lrf_speaker_on();
			delay(100);
			
			lrf_speaker_set_frequency(5000);
			delay(1000);
			
			lrf_speaker_off();
			delay(100);
			
			lrf_touch_read(&tempB);

			if(keyboard){ if(Serial.available()) { Serial.read(); passed = true; } }
			else		{ if(tempB != 0) passed = true; }
		}
		if(passed) Serial.println(" passed.");
		
		if(keyboard == false) done = true;
		else
		{
			Serial.print("[q] quit, [any] repeat");
			while(Serial.available()==false) delay(10);
			
			tempA = Serial.read();
			if(tempA == 'q') done = true;
		}
	}
	
	// ------------------------------------------- Everything
	Serial.print("everything has passed!!");
	
	// clear the test flag
	lrf_memory_write(LRF_MEMORY_HARDWARE_TEST_ADDR, LRF_MEMORY_HARDWARE_TEST_VALUE);
	
	// infinite loop
	while(true) delay(10);
}


#if TEST_SPEAKER
void lrf_speaker_test(void)
{
	unsigned int freq = 0;
	
	Serial.println("Testing speaker");
	lrf_speaker_on();
	while(freq < 5000)
	{
		lrf_speaker_set_frequency(freq);
		freq += 10;
		delay(10);
	}
	lrf_speaker_off();
}
#endif

#if TEST_RGB
void lrf_rgb_test(void)
{
	unsigned char val;
	
	for(val=0; val<LRF_RGB_VALUE_MAX; val++)
	{
		lrf_rgb_set(LRFRgbMask_LeftRed,val);
		delay(50);
	}
	rgb_set(LRFRgbMask_LeftRed,0);
	
	for(val=0; val<LRF_RGB_VALUE_MAX; val++)
	{
		lrf_rgb_set(LRFRgbMask_LeftGreen,val);
		delay(50);
	}
	rgb_set(LRFRgbMask_LeftGreen,0);
	
	for(val=0; val<LRF_RGB_VALUE_MAX; val++)
	{
		lrf_rgb_set(LRFRgbMask_LeftBlue,val);
		delay(50);
	}
	rgb_set(LRFRgbMask_LeftBlue,0);
	
	for(val=0; val<LRF_RGB_VALUE_MAX; val++)
	{
		lrf_rgb_set(LRFRgbMask_RightRed,val);
		delay(50);
	}
	rgb_set(LRFRgbMask_RightRed,0);
	
	for(val=0; val<LRF_RGB_VALUE_MAX; val++)
	{
		lrf_rgb_set(LRFRgbMask_RightGreen,val);
		delay(50);
	}
	rgb_set(LRFRgbMask_RightGreen,0);
	
	for(val=0; val<LRF_RGB_VALUE_MAX; val++)
	{
		lrf_rgb_set(LRFRgbMask_RightBlue,val);
		delay(50);
	}
	rgb_set(LRFRgbMask_RightBlue,0);
	
	for(val=0; val<LRF_RGB_VALUE_MAX; val++)
	{
		lrf_rgb_set(LRFRgbMask_LeftAll,val);
		delay(50);
	}
	rgb_set(LRFRgbMask_LeftAll,0);
	
	for(val=0; val<LRF_RGB_VALUE_MAX; val++)
	{
		lrf_rgb_set(LRFRgbMask_RightAll,val);
		delay(50);
	}
	rgb_set(LRFRgbMask_RightAll,0);
	
	for(val=0; val<LRF_RGB_VALUE_MAX; val++)
	{
		lrf_rgb_set(LRFRgbMask_All,val);
		delay(50);
	}
	lrf_rgb_set(LRFRgbMask_All,0);
}
#endif

#if TEST_TOUCH
void lrf_touch_test(void)
{
	unsigned char state=0;
	unsigned char count=0;
	
	Serial.print("Testing touch... ");
	delay(1000);
	
	while(count < 20)
	{
		lrf_touch_read(&state);
		//		if(state) Serial.print("1");
		//		else Serial.print("0");
		count++;
		delay(300);
	}
	
	//	Serial.println("done!");
	//	rgb_blink(LRFRgbMask_RightGreen);
}
#endif

