//
//  LRFDebug.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 2014-03-28.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#include "LRFDebug.h"
#include "LRFExpressions.h"
#include "LRFSensors.h"
#include "LRFEvents.h"
#include "LRFPersonality.h"
#include "LRFGames.h"
#include "LRFHardware.h"
#include "LRFMemory.h"

typedef enum
{
	LRFDebugConsoleMode_Menu = 0,
	LRFDebugConsoleMode_Expressions,
	LRFDebugConsoleMode_SpeechExpressions,
	LRFDebugConsoleMode_EyesExpressions,
	LRFDebugConsoleMode_Games,
	LRFDebugConsoleMode_Sensors,
	LRFDebugConsoleMode_Chatter,
	LRFDebugConsoleMode_Personality,
	LRFDebugConsoleMode_HardwareTest
} LRFDebugConsoleMode;

//------------------------------------------------------------
#pragma mark - Debug
//------------------------------------------------------------

void lrf_debug_stall(void)
{
	while(true)
	{
		Serial.println("stalled.");
		delay(1000);
	}
}


#if LRF_DEBUG_CONSOLE_ENABLED

void lrf_debug_personality_print(unsigned char trait)
{
	if(trait > 5)
	{
		Serial.print("???");
		return;
	}
	
	if(trait == 0)			Serial.print("brave");
	else if(trait == 1)		Serial.print("happy");
	else if(trait == 2)		Serial.print("wild");
	else if(trait == 3)		Serial.print("intro");
	else if(trait == 4)		Serial.print("quiet");
	else if(trait == 5)		Serial.print("funny");

	unsigned char *val = &lrfPersonality.brave + trait;
	Serial.print("(");
	Serial.print(*val);
	Serial.print(")");
}

void lrf_debug_console(void)
{
	LRFDebugConsoleMode mode = LRFDebugConsoleMode_Menu;
	unsigned int timer;
	
	while(true)
	{
		timer = 1000;
		
		// ignore incoming bytes
		while(Serial.available()) Serial.read();
		
		if(mode == LRFDebugConsoleMode_Menu)
		{
			// terminal prompt
			Serial.println("Select Test: ");
			Serial.println("[a] Expressions");
			Serial.println("[b] Speech");
			Serial.println("[c] Eyes");
			Serial.println("[d] Games");
			Serial.println("[e] Sensors");
			Serial.println("[f] Chatter");
			Serial.println("[g] Personality");
			Serial.println("[h] Hardware Test");
		}
		else if(mode == LRFDebugConsoleMode_Expressions)
		{
			Serial.print("Enter combo expression");
		}
		else if(mode == LRFDebugConsoleMode_SpeechExpressions)
		{
			Serial.print("Enter speech expression");
		}
		else if(mode == LRFDebugConsoleMode_EyesExpressions)
		{
			Serial.print("Enter eyes expression");
		}
		else if(mode == LRFDebugConsoleMode_Games)
		{
			Serial.print("Enter game");
		}
		else if(mode == LRFDebugConsoleMode_Sensors)
		{
			Serial.print("Which sensor? a=touch, b=light, c=mic, d=irin");
		}
		else if(mode == LRFDebugConsoleMode_Chatter)
		{
			Serial.print("Any key will chatter");
		}
		else if(mode == LRFDebugConsoleMode_Personality)
		{
			Serial.println("Which trait? a=brave, b=happy, c=wild, d=intro, e=quiet, f=funny");
			Serial.print("s=save, z=selector");
		}
		else if(mode == LRFDebugConsoleMode_HardwareTest)
		{
			Serial.print("Hardware testing...");
			lrf_hardware_test(true);
			mode = LRFDebugConsoleMode_Menu;
			continue;
		}
		Serial.println(" [Q] Quit");
		Serial.print("> ");
		
		// loop until input
		while(Serial.available()==0)
		{
			delay(10);
			timer--;
			if(timer == 0)
			{
				Serial.println("Timeout");
				return;
			}
		}
		
		// capture input (one byte)
		unsigned char input = Serial.read();
		unsigned char idx = input - 'a';
		bool bad = false;
		
		// echo input
		Serial.print((char)input);
		Serial.print(" (");
		Serial.print(input,DEC);
		Serial.println(")");
		
		if(input == 'Q')
		{
			if(mode != LRFDebugConsoleMode_Menu) mode = LRFDebugConsoleMode_Menu;
			else return;
		}
		
		if(mode == LRFDebugConsoleMode_Menu)
		{
			unsigned char menuItem = idx+1;
			if(menuItem >= LRFDebugConsoleMode_Expressions &&
			   menuItem <= LRFDebugConsoleMode_HardwareTest)
				mode = (LRFDebugConsoleMode)menuItem;
			else
				bad = true;
		}
		else if(mode == LRFDebugConsoleMode_Expressions)
		{
			if(idx >= 0 && idx < LRF_EXPRESSIONS_MAX_EXPRESSIONS) lrf_expressions_load_and_block((LRFExpression)idx);
			else bad = true;
		}
		else if(mode == LRFDebugConsoleMode_SpeechExpressions)
		{
//			if(idx >= 0 && idx < LRF_EXPRESSIONS_MAX_EXPRESSIONS)
//				;
////				lrf_speech_load_expression((LRFSoundExpressionName)idx);
//			else
//				bad = true;
		}
		else if(mode == LRFDebugConsoleMode_EyesExpressions)
		{
//			if(idx >= 0 && idx < LRF_EYES_MAX_EXPRESSIONS)
//				;
////				lrf_eyes_load_expression((LRFEyesExpressionName)idx);
//			else
//				bad = true;
		}
		else if(mode == LRFDebugConsoleMode_Games)
		{
			if(idx == 0)		lrf_games_taptaptap();
			else if(idx == 1)	lrf_games_songtime();
			else if(idx == 2)	lrf_games_peekaboo();
			else				bad = true;
		}
		else if(mode == LRFDebugConsoleMode_Sensors)
		{
			if(idx == 0)		lrf_touch_debug();
			else if(idx == 1)	lrf_light_debug();
			else if(idx == 2)	lrf_mic_debug();
			else				bad = true;
		}
		else if(mode == LRFDebugConsoleMode_Chatter)
		{
			lrf_expressions_chatter();
		}
		else if(mode == LRFDebugConsoleMode_Personality)
		{
			if(idx >= 0 && idx <= 5)
			{
				Serial.print("Selected ");
				lrf_debug_personality_print(idx);
				Serial.print(" ...value? a(1)-e(5):");
				while(Serial.available()==0) delay(1);
				unsigned char val = Serial.read() - 'a' + 1;
				if(val >= 1 && val <= 5)
				{
					Serial.println(val,DEC);
					Serial.print("Setting trait ");
					lrf_debug_personality_print(idx);
					Serial.print(" to ");
					Serial.print(val);
					Serial.println();
					
					unsigned char *ptr = &lrfPersonality.brave;
					ptr[idx] = val;
					
				} else bad = true;
			}
			else if(input == 'z')
			{
				lrf_personality_selection();
			}
			else if(input == 's')
			{
				lrf_memory_save();
			}
			else bad = true;
		}
		
		if(bad) Serial.println("bad entry");
	}
}

#endif

void lrf_debug_tag(const char *title, const char *tag, bool newline)
{
	Serial.print("[");
	Serial.print(title);
	Serial.print("] ");
	Serial.print(tag);
	if(newline) Serial.println();
}

void lrf_debug_pair(unsigned char a, unsigned char b, bool newline)
{
	Serial.print(a,DEC);
	Serial.print(",");
	Serial.print(b,DEC);
	if(newline) Serial.println();
}

void lrf_debug_list(const char *tag, unsigned char *list, unsigned char len, bool newline)
{
	Serial.print(tag);
	Serial.print("{");
	while(len--)
	{
		Serial.print(*list);
		list++;
		if(len) Serial.print(",");
	}
	Serial.print("}");
	if(newline) Serial.println();
}



