//
//  LRF.h
//  LittleRobotFriends
//
//  Created by Mark Argo on 2014-05-20.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

/*!
 
 \file LRF.h
 \brief This is the core API class for access LRF library functions.
 
 */

/*! \class LittleRobotFriend
	\brief The main class of the Little Robot Friends API
 
	This class is the main class of the LRF API. By importing LRF.h into your code, you are
	automatically creating an instance of the LRF library stored in the variable `lrf`.
 
	All functions listed in this document can be accessed using that static instance, eg:
 
	~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~{.c}
	lrf.blink(myPattern)
	lrf.say(mySound)
	~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 
 */

#ifndef __LRF_h
#define __LRF_h

#include <Arduino.h>
#include "LRFUtils.h"
#include "LRFEvents.h"
#include "LRFInfrared.h"

#define LRF_FIRMWARE_MAJOR						1	//! \def Something Once
#define LRF_FIRMWARE_MINOR						1	//! \def Something Else

#define LRF_DEBUG_CONSOLE_ENABLED				0
#define LRF_FORCE_HARDWARE_TEST					0
#define LRF_ADVANCED_HARDWARE_TEST				0


class LittleRobotFriend
{
private:
	
public:
	void setup(void);						//!< setup the LRF library
	void loop(void);						//!< run the LRF library loop
	
	//! @{
	//! \name Expressions (Lights and Sounds)
	
	/*! \fn		void blink(LRFPattern pattern)
		\brief	Blink a light pattern on the LED eyes
		\param	pattern Light pattern data object
		\param	duration Duration of the pattern in milliseconds
		\see	LRFPattern
		\see	LRFPatternStruct
		\see	LRFPatternMode
		\see	LRFColor
		
		This function causes your LRF’s LED eyes to blink a pattern. This is what is called a ‘blocking function’, meaning that no other code will run while your robot is blinking.
	 
		Usage:
	 
			LRFPattern myPattern = { LRFColor_Red, LRFColor_Green, LRFPatternMode_Fade };	// make a pattern
			lrf.blink(myPattern);		// blink the pattern for 1s (default)
			lrf.blink(myPattern, 5000)	// blink the pattern for 5s
			Serial.println("Pattern done!");
	*/
	void blink(LRFPattern pattern);
	void blink(LRFPattern pattern, unsigned int duration);
	 
	/*! \fn		void say(LRFSound sound)
		\brief	Speak a single sound
		\param	sound Sound data object
		\see	LRFSound
		\see	LRFSoundStruct
		\see	LRFNote
		\see	LRFOctave
		\see	LRFIntonation
		\see	LRFDuration
	 
		This function causes your LRF’s speaker to say a sound. This is what is called a ‘blocking function’, meaning that no other code will run while your robot is speaking.
	 
		Usage:
	
			LRFSound mySound = { LRFNote_C, LRFOctave_3, LRFIntonation_Rising, LRFDuration_Medium, LRFDuration_None };	// make a sound
			lrf.say(mySound);	// speak the sound
			Serial.println("Sound done!");
	 */
	void say(LRFSound sound);
	
	/*! \fn		void blinkAndSay(LRFPattern pattern, LRFSound sound)
		\brief	Blink and speak at the same time
		\param	pattern Light pattern data object
		\param	sound Sound data object
		\see	LRFPattern
		\see	LRFPatternStruct
		\see	LRFSound
		\see	LRFSoundStruct

		This function causes your LRF to blink and speak at the same time. this is a ‘blocking function’, meaning that no other code will run while your robot is expressing.
	 
		Usage:
		
			LRFPattern myPattern = { LRFColor_Red, LRFColor_Green, LRFPatternMode_Fade };	// make a pattern
			LRFSound mySound = { LRFNote_C, LRFOctave_3, LRFIntonation_Rising, LRFDuration_Medium, LRFDuration_None };	// make a sound
			lrf.blinkAndSay(myPattern, mySound);
			Serial.println("Blinkin' and speakin' done!");
	 */
	void blinkAndSay(LRFPattern pattern, LRFSound sound);
	
	/*! \fn		void blinkAndSay(LRFPattern pattern, LRFSound *sounds, char soundCount)
		\brief	Blink and speak at the same time
		\param	pattern Light pattern data object
		\param	sounds Pointer to array of sound data objects
		\param	soundCount Number of sounds in array
		\see	LRFPattern
		\see	LRFPatternStruct
		\see	LRFSound
		\see	LRFSoundStruct
	 
		This function causes your LRF to blink and speak multiple sounds at the same time. this is a ‘blocking function’, meaning that no other code will run while your robot is expressing. 
	 
		You may ask: "Why are there multiple sounds but only one pattern?". After each sound the pattern is 'reset', which means that in some instances it will go back to the same place, except with 'flip' patterns (eg: LRFPatternMode_FadeFlip), which flip the starting and target colors upon reset. If you really want to mix up your patterns, choose one with random colors (eg: LRFPatternMode_RandomFade).

		Usage:

			LRFPattern myPattern = { LRFColor_Blue, LRFColor_Pink, LRFPatternMode_FadeFlip };	// make a pattern
			LRFSound mySounds[3] = {
				{ LRFNote_C, LRFOctave_3, LRFIntonation_Rising, LRFDuration_Medium, LRFDuration_Short };
				{ LRFNote_D, LRFOctave_3, LRFIntonation_Rising, LRFDuration_Medium, LRFDuration_Medium };
				{ LRFNote_F, LRFOctave_3, LRFIntonation_Falling, LRFDuration_Long, LRFDuration_None };
			}; // make your array of sounds
			lrf.blinkAndSay(myPattern, mySounds, 3);
			Serial.println("Blinkin' and speakin' done!");
	 */
	void blinkAndSay(LRFPattern pattern, LRFSound *sounds, char soundCount);
	//! @}
	
	//! @{
	//! \name Events

	/*! \fn		void setEventHandler(LRFEvent event, LRFEventHandler handler)
		\brief	Customize how your robot responds to particular events.
		\param	event Event type constant
		\param	handler Event handler function pointer
		\see	LRFEvent
		\see	LRFEventHandler
	 
		This function allows you to customize how your Little Robot Friend responds to different events. Create a custom event handler function, fill it with code that shows a response, then attach it to a particular event. Easy!
	 
		Usage:
	 
			void myTapEventHandler(void) 	// create a custom event handler
			{
				lrf.blinkAndSay(myPattern, mySounds, 3);	// blink and say your unique expression
			}

			lrf.setEventHandler(LRFEvent_Tap, &myTapEventHandler);	// attach your handler to the event (you need the ampersand before the handler to make it a pointer

		With this example every time you 'tap' your LRF, it will call your custom function!
	 */
	void setEventHandler(LRFEvent event, LRFEventHandler handler);
	//! @}
	
	
	//!	@{ \name System
	 
	/*!	\fn		void setBoredom(unsigned int timeout, unsigned char count)
		\brief	Set how quickly and how often your robot gets bored.
		\param	timeout	Length of time in seconds between yawns (lower == bores quickly)
		\param	count	Number of yawns before falling to sleep (lower == sleeps quickly)
	 
		This function allows you to set how quickly your robot gets bored and falls asleep. In typical operation, the LRF gets bored when no interaction is detected for around 30-60 seconds (depending on personality). When this happens it 'yawns'. You may notice the sound it makes when it gets bored. After it's yawned 2-4 times (depending on personality) it finally goes to sleep. Now perhaps you want your robot to go to sleep when there is a particular interaction, or you want your robot to never sleep. Here is how:
	 
				lrf.setBoredom(999, 200);	// high numbers means no yawning, no sleep
				lrf.setBoredom(10, 1);		// low numbers means quick yawning (yawn in 10s, sleep after 1 yawn)
	 */
	void setBoredom(unsigned int timeout, unsigned char count);
	
	/*!	\fn		void sleep(void)
		\brief	Put the robot to sleep
	 
		Things.
	 */
	void sleep(void);
	//! @}

	
	//!	@{ \name Infrared

	/*!	\fn		LRFIRMessage readIRMessage(void)
		\brief	Read the latest received IR message
		\return	3-bit LRFIRMessage constant
	 
		Things.
	 */
	LRFIRMessage readIRMessage(void);

	/*!	\fn		void sendIRMessage(LRFIRMessage message)
		\brief	Send an IR message
		\param	message	3-bit LRFIRMessage constant
	 
		Things.
	 */
	void sendIRMessage(LRFIRMessage message);
	//! @}

	//!	@{ \name Sensors

	/*!	\fn		unsigned char readTouch(void)
		\brief	Read the touch sensor
		\return	Whether or not the LRF is being touched (1 or 0)
	 
		Things.
	 */
	unsigned char readTouch(void);

	/*!	\fn		unsigned char readLight(void)
		\brief	Read the light sensor
		\return	Raw value of the sensor (higher == brighter)
	 
		Things.
	 */
	unsigned char readLight(void);

	/*!	\fn		unsigned char readMicrophone(void)
		\brief	Read the microphone
		\return	Raw value of the microphone (higher == louder)
	 
		Things.
	 */
	unsigned char readMicrophone(void);
	//! @}
	
	//!	@{ \name Outputs

	/*!	@{
	 \brief	Set the RGB value of one or both LEDs
	 \param	red		Red value (max = 64)
	 \param	green	Green value (max = 64)
	 \param	blue	Blue value (max = 64)
	 \param	update	Whether or not to update the LEDs after setting
	 
	 Things.
	 */
	void setLeftLED(unsigned char red, unsigned char green, unsigned char blue, bool update=true);
	void setRightLED(unsigned char red, unsigned char green, unsigned char blue, bool update=true);
	void setBothLEDs(unsigned char red, unsigned char green, unsigned char blue, bool update=true);
	//! @}
	
	/*!	\fn		void sendIRMessage(LRFIRMessage message)
	 \brief	Send an IR message
	 \param	message	3-bit LRFIRMessage constant
	 
	 Things.
	 */
	void setSpeaker(unsigned int frequency, unsigned int duration);
	//! @}
	
};

extern LittleRobotFriend lrf;

#endif