//
//  LRFEvents.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 2014-03-14.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#include "LRFEvents.h"
#include "LRFPersonality.h"
#include "LRFDebug.h"

const char debug_title[] = "evt";

LRFEvents lrfEvents = {
	true,
	{
		LRFEvent_Unknown,
		LRFEvent_Unknown,
		LRFEvent_Unknown,
		LRFEvent_Unknown
	},
	{
		LRFEvent_Unknown,
		LRFEvent_Unknown,
		LRFEvent_Unknown,
		LRFEvent_Unknown
	},
	0, 0, 0, 0
};


void lrf_events_enable(void)
{
#if LRF_EVENTS_DEBUG_VERBOSE
	lrf_debug_tag(debug_title, "e");
#endif

	lrfEvents.enabled = true;
}

void lrf_events_disable(void)
{
#if LRF_EVENTS_DEBUG_VERBOSE
	lrf_debug_tag(debug_title, "d");
#endif
	
	lrfEvents.enabled = false;
}

void lrf_events_update_interaction(void)
{
	lrfEvents.lastInteractionTime = millis();
	lrfEvents.boredTimeout = lrfEvents.lastInteractionTime + lrfEvents.boredTimeoutReload;
	lrfEvents.boredCount = 0;
}

void lrf_events_setup(void)
{
	lrf_events_update_interaction();
	
	lrfEvents.boredTimeoutReload = LRF_BORED_TIME;
	lrfEvents.boredTimeoutReload += 5000 * ((signed char)(LRF_PERSONALITY_LEVEL_NORMAL - lrfPersonality.neediness));
	
	lrfEvents.boredCountReload = LRF_BORED_TO_SLEEP_COUNT;
	if(lrfPersonality.neediness == LRF_PERSONALITY_LEVEL_HIGH)		lrfEvents.boredCountReload++;
	else if(lrfPersonality.neediness == LRF_PERSONALITY_LEVEL_LOW)	lrfEvents.boredCountReload--;
}

void lrf_events_process(void)
{
	// is event processing enabled?
	if(lrfEvents.enabled)
	{
		// if we have any events in our queue...
		if(lrfEvents.eventQueueCount > 0)
		{
			LRFEvent nextEvent = lrfEvents.eventQueue[0];
			
			// call the next event
			lrfEventHandlers[nextEvent]();
			
			// add non-system events to history
			if(nextEvent < LRF_EVENTS_SYSTEM_START_IDX)
			{
				// cycle the history
				for(unsigned char i=(LRF_EVENTS_HISTORY_SIZE-1); i>0; i--)
				{
					lrfEvents.eventHistory[i] = lrfEvents.eventHistory[i-1];
				}
				
				// mark event as the last one called
				lrfEvents.eventHistory[0] = nextEvent;
			}
			
			// decrement our queue
			lrfEvents.eventQueueCount--;
			
			// cycle the queue
			for(unsigned char i=0; i<lrfEvents.eventQueueCount; i++)
			{
				lrfEvents.eventQueue[i] = lrfEvents.eventQueue[i+1];
			}
		}

#if LRF_BORED_DISABLE == 0
		if(millis() > lrfEvents.boredTimeout)
		{
			lrfEvents.boredCount++;
			lrfEvents.boredTimeout = millis() + lrfEvents.boredTimeoutReload;
			
#if LRF_EVENTS_DEBUG
			lrf_debug_tag(debug_title, "bored");
#endif
			
			if(lrfEvents.boredCount >= lrfEvents.boredCountReload)
			{
				// sleep
				lrf_events_add(LRFEvent_Sleep);
			}
			else
			{
				// bored
				lrf_events_add(LRFEvent_IsBored);
			}
		}
#endif
	}
}

void lrf_events_add(LRFEvent event)
{
	// if we have space in our event queue...
	if(lrfEvents.eventQueueCount < LRF_EVENTS_QUEUE_SIZE)
	{
		
#if LRF_BORED_EVENT_ADD_FILTER
		// filter out everything but tickles and hugs when bored
		if(lrfEvents.boredCount > 0 && !(event == LRFEvent_Sleep || event == LRFEvent_IsBored || event == LRFEvent_Tickle || event == LRFEvent_Hug || event == LRFEvent_MessageReceived))
		{
			//return;
			event = LRFEvent_IsBored;
		}
#endif

		// add the event to our queue and increment count
		lrfEvents.eventQueue[lrfEvents.eventQueueCount] = event;
		lrfEvents.eventQueueCount++;
		
		// reset our interaction timer if it's a non-system event
		if(event < LRF_EVENTS_SYSTEM_START_IDX || event == LRFEvent_PowerUp)
		{
			lrf_events_update_interaction();
		}
		
#if LRF_EVENTS_DEBUG_VERBOSE
		lrf_debug_tag(debug_title, "a:", false);
		Serial.println(event,DEC);
#endif
	}
}

void lrf_events_force(LRFEvent event)
{
	// whoa!
#if LRF_EVENTS_DEBUG_VERBOSE
	lrf_debug_tag(debug_title, "f:", false);
	Serial.println(event,DEC);
#endif
	lrfEventHandlers[event]();
}

// how often has this event come up in our history?
unsigned char lrf_events_history_count(LRFEvent event)
{
	unsigned char count = 0;
	for(unsigned char i=0; i<LRF_EVENTS_HISTORY_SIZE; i++)
	{
		if(lrfEvents.eventHistory[i] == event) count++;
	}
	return count;
}

// how many times has our event come up in a row?
unsigned char lrf_events_history_continuous(LRFEvent event)
{
	unsigned char count = 0;
	for(unsigned char i=0; i<LRF_EVENTS_HISTORY_SIZE; i++)
	{
		if(lrfEvents.eventHistory[i] == event) count++;
		else break;
	}
	return count;
}



