//
//  LRFSpeech.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 1/14/2014.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#include "LRFSpeech.h"
#include "LRFHardware.h"
#include "LRFPersonality.h"
#include "LRFPower.h"

#if LRF_SPEECH_DEBUG
#include "LRFDebug.h"
const char debug_title[] = "sp";
#endif

//-------------------------------------------------------------
#pragma mark - Variables
//-------------------------------------------------------------

LRFSpeech lrfSpeech = { LRFSpeechState_Idle };



//-------------------------------------------------------------
#pragma mark - Reset
//-------------------------------------------------------------

//inline void lrf_speech_done_expression(void)
//{
//	lrfSpeech.state = LRFSpeechState_Idle;
//	lrf_speaker_off(); // just-in-case
//	if(lrfSpeech.doneFunction != 0) lrfSpeech.doneFunction();
//}

void lrf_speech_reset(void)
{
	lrfSpeech.state = LRFSpeechState_Idle;
	lrf_speaker_off();
}


//-------------------------------------------------------------
#pragma mark - Personality
//-------------------------------------------------------------

void lrf_speech_apply_personality(void)
{
	// note: would love to find a better way to do this??
	// note: sound should be loaded, with duration looked up!
	
	signed char octave = lrfSpeech.sound.octave;
	signed int duration = lrfSpeech.duration;

#if LRF_SPEECH_DEBUG
	lrf_debug_tag(debug_title, "before", false);
	Serial.print(" oct:");
	Serial.print((unsigned char)lrfSpeech.sound.octave,DEC);
	Serial.print(" dur:");
	Serial.println((unsigned int)lrfSpeech.duration,DEC);
#endif
	
	// shift octave
	octave += ((signed char)lrfPersonality.enthusiasm - LRF_PERSONALITY_LEVEL_NORMAL);
	if(octave < LRFOctave_2)		octave = LRFOctave_2;
	else if(octave > LRFOctave_7)	octave = LRFOctave_7;
	lrfSpeech.sound.octave = (LRFOctave)octave;
	
	// shift duration
	duration *= (0.2 * ((signed char)LRF_PERSONALITY_LEVEL_NORMAL-lrfPersonality.enthusiasm));
	lrfSpeech.duration += duration;

#if LRF_SPEECH_DEBUG
	lrf_debug_tag(debug_title, "after", false);
	Serial.print(" oct:");
	Serial.print((unsigned char)lrfSpeech.sound.octave,DEC);
	Serial.print(" dur:");
	Serial.println((unsigned int)lrfSpeech.duration,DEC);
#endif
}

//-------------------------------------------------------------
#pragma mark - Playback
//-------------------------------------------------------------

void lrf_speech_play_sound_and_block(LRFSound sound, bool applyPersonality)
{
	// load the sound data
	lrf_speech_play_sound(sound, applyPersonality);
	
	// block until the sound is done
	while(lrfSpeech.state != LRFSpeechState_Done)
	{
		lrf_speech_process();
		delay(5);
	}
}

void lrf_speech_play_sound(LRFSound sound, bool applyPersonality)
{
	// note: does it matter if we're already playing?
	
	// copy the sound structure from the union to a local copy
	lrfSpeech.sound = sound.sound;
	
	// lookup duration variables
	lrfSpeech.duration = lrf_utils_duration_lookup(lrfSpeech.sound.duration);
	
	// shift our values based on personality
	if(applyPersonality) lrf_speech_apply_personality();
	
	// look-up the frequency
	lrfSpeech.frequency = lrf_utils_note_to_frequency(lrfSpeech.sound.note, lrfSpeech.sound.octave);

	// update our state so we begin playing on the next process call
	lrfSpeech.state = LRFSpeechState_Ready;
}

void lrf_speech_process()
{
	// let's quickly bail if we're not speaking
	if(lrfSpeech.state == LRFSpeechState_Idle || lrfSpeech.state == LRFSpeechState_Done)
	{
		return;
	}
	
	unsigned long now = millis();
	
	// if we're ready, then let's turn on our speaker and load our frequency
	if(lrfSpeech.state == LRFSpeechState_Ready)
	{
		// mark the start time
		lrfSpeech.startTime = now;

		// begin speaking!
		if(lrfPower.isHungry == false) lrf_speaker_on();
		lrf_speaker_set_frequency(lrfSpeech.frequency);
		
		// update our state to speaking
		lrfSpeech.state = LRFSpeechState_Speaking;
	}
	
	// otherwise we're speaking...
	else if(lrfSpeech.state == LRFSpeechState_Speaking)
	{
		// if we've completed our sound...
		if(now > lrfSpeech.startTime + lrfSpeech.duration)
		{
			// turn off the speaker & set up for our pause time (if necessary)
			lrf_speaker_off();
			
			// check to see if we even have a duration
			if(lrfSpeech.sound.pause == LRFDuration_None)
			{
				lrfSpeech.state = LRFSpeechState_Done;
			}
			else
			{
				lrfSpeech.startTime = now;
				lrfSpeech.duration = lrf_utils_duration_lookup(lrfSpeech.sound.pause);
				
				// update our state to show that we're pausing
				lrfSpeech.state = LRFSpeechState_Pausing;
			}
		}
		
		// we're still playing our sound...
		else
		{
			// if our intonation isn't flat, we will need to adjust our frequency
			if(lrfSpeech.sound.intonation == LRFIntonation_Flat)
			{
				return;
			}
			else if(lrfSpeech.sound.intonation == LRFIntonation_Rising)
			{
				if(lrfSpeech.frequency > 0) lrfSpeech.frequency--;
			}
			else if(lrfSpeech.sound.intonation == LRFIntonation_Falling)
			{
				if(lrfSpeech.frequency < 0xffff) lrfSpeech.frequency++;
			}
			else if(lrfSpeech.sound.intonation == LRFIntonation_Peaking)
			{
				if(now > (lrfSpeech.startTime + lrfSpeech.duration/2))
				{
					if(lrfSpeech.frequency < 0xfffe) lrfSpeech.frequency += 2;
				}
				else
				{
					if(lrfSpeech.frequency > 2) lrfSpeech.frequency -= 2;
				}
			}
			else if(lrfSpeech.sound.intonation == LRFIntonation_Dipping)
			{
				if(now > (lrfSpeech.startTime + lrfSpeech.duration/2))
				{
					if(lrfSpeech.frequency > 2) lrfSpeech.frequency -= 2;
				}
				else
				{
					if(lrfSpeech.frequency < 0xfffe) lrfSpeech.frequency += 2;
				}
			}

			lrf_speaker_set_frequency(lrfSpeech.frequency);
		}
	}
	
	// ... or we're pausing
	else if(lrfSpeech.state == LRFSpeechState_Pausing)
	{
		// if we've completed our pause...
		if(now > lrfSpeech.startTime + lrfSpeech.duration)
		{
			lrfSpeech.state = LRFSpeechState_Done;
		}
	}
}

