//
//  LRFSensors.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 1/16/2014.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#include "LRFSensors.h"
#include "LRFHardware.h"
#include "LRFEvents.h"
#include "LRFPersonality.h"

#if LRF_SENSORS_DEBUG
#include "LRFDebug.h"
const char debug_title[] = "sns";
#if LRF_MIC_DEBUG
const char mic_debug_title[] = "mic";
#endif
#if LRF_LIGHT_DEBUG
const char light_debug_title[] = "lgt";
#endif
#if LRF_TOUCH_DEBUG
const char touch_debug_title[] = "tch";
#endif
#endif



LRFSensors lrfSensors = {
	true, 0, false
};

LRFSensorBuffer lightSmoother;
LRFSensorBuffer micSmoother;
LRFSensorBuffer micSamples;
unsigned char lightDarkValue = LRF_LIGHT_DARK_VALUE;


//------------------------------------------------------------
#pragma mark - Private Prototypes
//------------------------------------------------------------

void lrf_sensor_reset(LRFSensorStruct *sensor, bool buffReset=true);
void lrf_sensor_buffer_reset(LRFSensorBuffer *buffer);
unsigned char lrf_sensor_buffer_update(LRFSensorBuffer *buffer, unsigned char value);



//------------------------------------------------------------
#pragma mark - Touch
//------------------------------------------------------------

inline void lrf_touch_reset(void)
{
#if LRF_SENSORS_DEBUG
	lrf_debug_tag(debug_title, "tr");
#endif
	lrf_sensor_reset(&lrfSensors.touch);
}

inline void lrf_touch_process(void)
{
	// let's check our current timing
	if(millis() < lrfSensors.touch.nextReadTime) return;
	lrfSensors.touch.nextReadTime = millis() + LRF_TOUCH_READ_FREQUENCY;
	
	// read the sensor
	lrf_touch_read(&lrfSensors.touch.value);
	lrfSensors.touch.timestamp = millis() >> 8;
	
	// if current value doesn't match the last value
	if(lrfSensors.touch.value != lrfSensors.touch.last->value)
	{
		LRFSensorSample *s = &lrfSensors.touch.samples[lrfSensors.touch.idx];
		
		s->value = lrfSensors.touch.value;
		s->timestamp = lrfSensors.touch.timestamp;
		
#if LRF_TOUCH_DEBUG
		Serial.print(s->value, DEC);
		Serial.print(" ");
		Serial.print(s->timestamp, DEC);
		Serial.println();
#endif
		
		lrfSensors.touch.last = s;
		lrfSensors.touch.idx = (lrfSensors.touch.idx + 1) % LRF_SENSOR_SAMPLE_COUNT;
		
		// only evaluate on falling signal
		if(lrfSensors.touch.last->value == 0)
		{
			lrfSensors.touch.evaluate = true;
		}
	}
	
	if(lrfSensors.touch.evaluate)
	{
		// wait ~500ms after low signal to read
		if(lrfSensors.touch.timestamp > lrfSensors.touch.last->timestamp + 1)
		{
			// how many touches in the past X seconds?
			unsigned char count = 0;
			unsigned char idx = lrfSensors.touch.idx; // index of oldest sample
			
			for(unsigned char i=0; i<LRF_SENSOR_SAMPLE_COUNT; i++)
			{
				// if less than our threshold time...
				if(lrfSensors.touch.samples[idx].value == 1)
				{
					if((lrfSensors.touch.timestamp - lrfSensors.touch.samples[idx].timestamp) < 8) count++;
				}
				idx = (idx+1) % LRF_SENSOR_SAMPLE_COUNT;
			}
			
#if LRF_TOUCH_DEBUG
			Serial.print("count:");
			Serial.print(count,DEC);
#endif
			
			if(count == 0)
			{
#if LRF_TOUCH_DEBUG
				unsigned char lastTouchIdx = (lrfSensors.touch.idx + (LRF_SENSOR_SAMPLE_COUNT-2)) % LRF_SENSOR_SAMPLE_COUNT;
				unsigned int len = lrfSensors.touch.last->timestamp - lrfSensors.touch.samples[lastTouchIdx].timestamp;
				len = len << 8;
				Serial.print(" hug:");
				Serial.print(len,DEC);
#endif
				lrf_events_add(LRFEvent_Hug);
			}
			else if(count == 1)
			{
#if LRF_TOUCH_DEBUG
				Serial.print(" tap!");
#endif
				lrf_events_add(LRFEvent_Tap);
			}
			else
			{
#if LRF_TOUCH_DEBUG
				Serial.print(" tickle!");
#endif
				lrf_events_add(LRFEvent_Tickle);
			}
			
#if LRF_TOUCH_DEBUG
			Serial.println();
#endif
			
			// done evaluation until next state change
			lrfSensors.touch.evaluate = false;
		}
	}
}

void lrf_touch_debug(void)
{
	lrf_touch_reset();
	
	while(true)
	{
		lrf_touch_process();
		if(Serial.available()) break;
	}
}

//------------------------------------------------------------
#pragma mark - Light
//------------------------------------------------------------

inline void lrf_light_reset(void)
{
#if LRF_SENSORS_DEBUG
	lrf_debug_tag(debug_title, "lr");
#endif
	lrf_sensor_buffer_reset(&lightSmoother);
	lrf_sensor_reset(&lrfSensors.light,false);
	lrfSensors.light.nextReadTime = millis() + (LRF_LIGHT_READ_FREQUENCY*4);
}

inline void lrf_light_process(void)
{
	// let's check our current timing
	if(millis() < lrfSensors.light.nextReadTime) return;
	lrfSensors.light.nextReadTime = millis() + LRF_LIGHT_READ_FREQUENCY;
	
	// add a sample to the smoothing buffer
	unsigned char val = lrf_sensor_buffer_update(&lightSmoother, (analogRead(7) >> 2));
	if(lightSmoother.ready == false)
	{
#if LRF_LIGHT_DEBUG
		lrf_debug_tag(light_debug_title, "!rdy");
#endif
		return;
	}
	
	lrfSensors.light.value = val;
	lrfSensors.light.timestamp = millis() >> 8;
	
#if LRF_LIGHT_DEBUG && LRF_LIGHT_DEBUG_VERBOSE
	Serial.print(lrfSensors.light.value, DEC);
	Serial.print(" ");
	Serial.print(lrfSensors.light.timestamp, DEC);
	Serial.print(" ");
	Serial.print(lrfSensors.light.last->value, DEC);
#endif
	
	// calibrate (if required)
	if(lrfSensors.light.calibrated == false)
	{
		// are we in a darker place?
		if(lrfSensors.light.value < 50)
		{
			lightDarkValue = LRF_LIGHT_DARK_VALUE - LRF_LIGHT_HISTERESYS_SIZE*2;
		}
		else if(lrfSensors.light.value < 100)
		{
			lightDarkValue = LRF_LIGHT_DARK_VALUE - LRF_LIGHT_HISTERESYS_SIZE;
		}
		// or are we in a light place?
		else if(lrfSensors.light.value > 180)
		{
			lightDarkValue = LRF_LIGHT_DARK_VALUE + LRF_LIGHT_HISTERESYS_SIZE;
		}
		else
		{
			lightDarkValue = LRF_LIGHT_DARK_VALUE;
		}
		
#if LRF_LIGHT_DEBUG
		Serial.print("calib:");
		Serial.print(lrfSensors.light.value,DEC);
		Serial.print("-");
		Serial.print(lightDarkValue,DEC);
		Serial.println();
#endif
		
		lrfSensors.light.calibrated = true;
	}
	
	// evaluate for our next state
	unsigned char nextState = LRF_LIGHT_STATE_UNKNOWN;
	if(lrfSensors.light.value <= (lightDarkValue - LRF_LIGHT_HISTERESYS_SIZE/2))
	{
		nextState = LRF_LIGHT_STATE_DARK;
	}
	else if(lrfSensors.light.value >= (LRF_LIGHT_BRIGHT_VALUE + LRF_LIGHT_HISTERESYS_SIZE/2))
	{
		nextState = LRF_LIGHT_STATE_BRIGHT;
	}
	else if(lrfSensors.light.value >= (lightDarkValue + LRF_LIGHT_HISTERESYS_SIZE/2) &&
			lrfSensors.light.value <= (LRF_LIGHT_BRIGHT_VALUE - LRF_LIGHT_HISTERESYS_SIZE/2))
	{
		nextState = LRF_LIGHT_STATE_NORMAL;
	}
	
	// is this state a change?
	if(nextState != LRF_LIGHT_STATE_UNKNOWN)
	{
		LRFSensorSample *s = &lrfSensors.light.samples[lrfSensors.light.idx];
		s->value = nextState;
		s->timestamp = lrfSensors.light.timestamp;
		
		if(nextState != lrfSensors.light.last->value)
		{
			
			if(nextState == LRF_LIGHT_STATE_DARK && lrfSensors.light.last->value != LRF_LIGHT_STATE_LONGDARK)
			{
#if LRF_LIGHT_DEBUG
				Serial.print("LIGHTSOFF!");
#endif
				lrf_events_add(LRFEvent_LightsOff);
			}
			else if(nextState == LRF_LIGHT_STATE_BRIGHT)
			{
#if LRF_LIGHT_DEBUG
				Serial.print("BRIGHT!");
#endif
				lrf_events_add(LRFEvent_LightsBright);
			}
			else if(lrfSensors.light.last->value == LRF_LIGHT_STATE_DARK && nextState != LRF_LIGHT_STATE_DARK)
			{
#if LRF_LIGHT_DEBUG
				Serial.print("LIGHTSON!");
#endif
				lrf_events_add(LRFEvent_LightsOn);
			}
			
			lrfSensors.light.last = s;
			lrfSensors.light.idx = (lrfSensors.light.idx + 1) % LRF_SENSOR_SAMPLE_COUNT;
		}
		else
		{
			if(nextState == LRF_LIGHT_STATE_DARK && lrfSensors.light.last->value == LRF_LIGHT_STATE_DARK)
			{
				unsigned char longtime = 25;
				longtime += ((signed char) LRF_PERSONALITY_LEVEL_NORMAL-lrfPersonality.bravery)*2;
				
				if((lrfSensors.light.timestamp - lrfSensors.light.last->timestamp) > longtime)
				{
#if LRF_LIGHT_DEBUG
					Serial.print("LONGDARK!!");
#endif
					lrf_events_add(LRFEvent_LightsLongDark);
					
					// add event here!
					s->value = LRF_LIGHT_STATE_LONGDARK;
					lrfSensors.light.last = s;
					lrfSensors.light.idx = (lrfSensors.light.idx + 1) % LRF_SENSOR_SAMPLE_COUNT;
				}
			}
		}
				
#if LRF_LIGHT_DEBUG && LRF_LIGHT_DEBUG_VERBOSE==0
		Serial.println();
#endif
	}
	
#if LRF_LIGHT_DEBUG && LRF_LIGHT_DEBUG_VERBOSE
	Serial.println();
#endif
	
}

void lrf_light_debug(void)
{
	lrf_light_reset();
	
	while(true)
	{
		lrf_light_process();
		if(Serial.available()) break;
	}
}


//------------------------------------------------------------
#pragma mark - Mic
//------------------------------------------------------------

inline void lrf_mic_reset(void)
{
#if LRF_SENSORS_DEBUG
	lrf_debug_tag(debug_title, "mr");
#endif
	lrf_sensor_buffer_reset(&micSmoother);
	lrf_sensor_buffer_reset(&micSamples);
	lrf_sensor_reset(&lrfSensors.mic);
}

inline void lrf_mic_process(void)
{
	// let's check our current timing
	if(millis() < lrfSensors.mic.nextReadTime) return;
	lrfSensors.mic.nextReadTime = millis() + LRF_MIC_READ_FREQUENCY;
	
	// read our value into our smoothing buffer
	unsigned char val = lrf_sensor_buffer_update(&micSmoother, (analogRead(6) >> 2));
	if(micSmoother.ready == false)
	{
#if LRF_MIC_DEBUG
		lrf_debug_tag(mic_debug_title, "!rdy");
#endif
		return;
	}
	
	// add smoothed value to our sample buffer
	unsigned char lastIdx = micSamples.idx;
	lrf_sensor_buffer_update(&micSamples, val);
	if(micSamples.ready == false)
	{
#if LRF_MIC_DEBUG
		lrf_debug_tag(mic_debug_title, "!rdy");
#endif
		return;
	}
		
#if LRF_MIC_DEBUG && LRF_MIC_DEBUG_VERBOSE
	while(val--) Serial.print("=");
	Serial.print("| ");
	Serial.print(micSamples.buffer[lastIdx], DEC);
#endif
			
//	// update our indexes
//	unsigned char lastIdx = micSamples.idx;
//	micSamples.idx = (micSamples.idx+1) % LRF_SENSOR_SAMPLE_COUNT;
//	if(!lrfSensors.mic.ready)
//	{
//		if(micSamples.idx==0)
//		{
//#if LRF_MIC_DEBUG
//			lrf_debug_tag(mic_debug_title, "rdy");
//#endif
//			lrfSensors.mic.ready=true;
//		}
//		else
//		{
//#if LRF_MIC_DEBUG
//			lrf_debug_tag(mic_debug_title, "!rdy");
//#endif
//			return;
//		}
//	}
	
	// get our difference between first & last index
	signed char diff = micSamples.buffer[lastIdx] - micSamples.buffer[micSamples.idx];
	lrfSensors.mic.value = LRF_MIC_STATE_NORMAL;
	lrfSensors.mic.timestamp = millis() >> 8;
	if(diff >= LRF_MIC_DIFF_THRESHOLD)			lrfSensors.mic.value = LRF_MIC_STATE_RISING;
	else if(diff <= -LRF_MIC_DIFF_THRESHOLD)	lrfSensors.mic.value = LRF_MIC_STATE_FALLING;
	
	// should we log this change?
	if(lrfSensors.mic.value != LRF_MIC_STATE_NORMAL && (lrfSensors.mic.timestamp-lrfSensors.mic.last->timestamp) > LRF_MIC_TIME_THRESHOLD)
	{
		LRFSensorSample *s = &lrfSensors.mic.samples[lrfSensors.mic.idx];
		s->value = lrfSensors.mic.value;
		s->timestamp = lrfSensors.mic.timestamp;
		
		lrf_events_add(LRFEvent_HeardLoudNoise);
		
#if LRF_MIC_DEBUG
		Serial.print(lrfSensors.mic.value,DEC);
		Serial.print(" ");
		Serial.print(lrfSensors.mic.timestamp,DEC);
		Serial.print(" ");
		if(lrfSensors.mic.value == LRF_MIC_STATE_RISING)		Serial.print("RISING");
		else if(lrfSensors.mic.value == LRF_MIC_STATE_FALLING)	Serial.print("FALLING");
#if LRF_MIC_DEBUG_VERBOSE==0
		Serial.println();
#endif
#endif
		
#if LRF_MIC_DEBUG && LRF_MIC_DEBUG_EVENT_BREAK
		lrf_debug_tag(mic_debug_title, "break");
		while(!Serial.available());
		while(Serial.available()) Serial.read();
#endif
		
		lrfSensors.mic.last = s;
		lrfSensors.mic.idx = (lrfSensors.mic.idx+1) % LRF_SENSOR_SAMPLE_COUNT;
	}
	
#if LRF_MIC_DEBUG && LRF_MIC_DEBUG_VERBOSE
	Serial.println();
#endif
}

void lrf_mic_debug(void)
{
	lrf_mic_reset();
	
	while(true)
	{
		lrf_mic_process();
		if(Serial.available()) break;
	}
}


//------------------------------------------------------------
#pragma mark - General Sensors
//------------------------------------------------------------

void lrf_sensors_setup(void)
{
	lrf_sensors_enable();
	lrf_sensors_recalibrate();
}

void lrf_sensors_recalibrate(void)
{
	lrfSensors.touch.calibrated = false;
	lrfSensors.light.calibrated = false;
	lrfSensors.mic.calibrated = false;
}

void lrf_sensors_process(void)
{
	if(lrfSensors.enabled == false)
	{
		if(lrfSensors.enableTimer > 0)
		{
			lrfSensors.enableTimer--;
			if(lrfSensors.enableTimer<10)
			{
#if LRF_SENSORS_DEBUG
				lrf_debug_tag(debug_title, "e++");
#endif
				lrfSensors.enabled = true;
			}
		}
		return;
	}

#if LRF_TOUCH_ENABLE
	lrf_touch_process();
#endif
		
#if LRF_MIC_ENABLE
	lrf_mic_process();
#endif
		
#if LRF_LIGHT_ENABLE
	lrf_light_process();
#endif
}

void lrf_sensors_enable(void)
{
#if LRF_SENSORS_DEBUG
	lrf_debug_tag(debug_title, "e+");
#endif
		
#if LRF_TOUCH_ENABLE
	lrf_touch_reset();
#endif
	
#if LRF_MIC_ENABLE
	lrf_mic_reset();
#endif
	
#if LRF_LIGHT_ENABLE
	lrf_light_reset();
#endif
	
	lrfSensors.enableTimer = 0x1fff;
	//lrfSensors.enabled = true;
}

void lrf_sensors_disable(void)
{
#if LRF_SENSORS_DEBUG
	lrf_debug_tag(debug_title, "e-");
#endif
	lrfSensors.enabled = false;
	lrfSensors.enableTimer = 0;
}

void lrf_sensor_reset(LRFSensorStruct *sensor, bool buffReset)
{
	sensor->value = 0;
	sensor->timestamp = 0;
	sensor->evaluate = false;
	sensor->nextReadTime = 0;
	if(buffReset)
	{
		sensor->idx = 0;
		sensor->last = &sensor->samples[0];
		for(unsigned char i=0; i<LRF_SENSOR_SAMPLE_COUNT; i++)
		{
			sensor->samples[i].value = 0;
			sensor->samples[i].timestamp = 0;
		}
	}
}

void lrf_sensor_buffer_reset(LRFSensorBuffer *buffer)
{
	for(buffer->idx=0; buffer->idx<LRF_SMOOTHING_BUFFER_COUNT; buffer->idx++)
	{
		buffer->buffer[buffer->idx] = 0;
	}
	buffer->idx = 0;
	buffer->ready = false;
}

unsigned char lrf_sensor_buffer_update(LRFSensorBuffer *buffer, unsigned char value)
{
	unsigned int mean = 0;
	
	// add the value
	buffer->buffer[buffer->idx] = value;
	buffer->idx = (buffer->idx+1) % LRF_SMOOTHING_BUFFER_COUNT;
	
	for(unsigned char i=0; i<LRF_SMOOTHING_BUFFER_COUNT; i++)
	{
		mean += buffer->buffer[i];
	}
	
	if(buffer->ready == false)
	{
		if(buffer->idx==0) buffer->ready = true;
		else return 0;
	}
	
	return (mean / LRF_SMOOTHING_BUFFER_COUNT);
}

