//
//  LRFInfrared.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 2014-03-13.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#include "LRFInfrared.h"
#include "LRFHardware.h"
#include "LRFEvents.h"
#include "LRFDebug.h"

#if LRF_IRDA_DEBUG
const char debug_title[] = "ir";
#endif

LRFIR lrfIrda = { false, false, LRFIRMessage_Unknown, LRFIRMessage_Unknown };

void lrf_irda_setup(void)
{
	// enable INT1 for reading
	attachInterrupt(1, lrf_irda_read_message, FALLING);
	lrf_irda_read_enable();
}

void lrf_irda_process(void)
{
	if(lrfIrda.hasMessage)
	{
		if(!lrfIrda.dispatchedMessage)
		{
#if LRF_IRDA_DEBUG
			lrf_debug_tag(debug_title, "in", false);
			Serial.println((unsigned char)lrfIrda.message,DEC);
#endif

			if(lrfIrda.message >= LRFIRMessage_Hello && lrfIrda.message <= LRFIRMessage_LeaveMeAlone)
			{
				lrf_events_add(LRFEvent_MessageReceived);
				lrfIrda.dispatchedMessage = true;
			}
			else
			{
				lrf_irda_reset_read();
			}
		}
	}
}

void lrf_irda_reset_read(void)
{
	lrfIrda.hasMessage = false;
	lrfIrda.dispatchedMessage = false;
	lrf_irda_read_enable();
}

//------------------------------------------------------------
#pragma mark - Queueing
//------------------------------------------------------------

void lrf_irda_queue_message(LRFIRMessage message)
{
#if LRF_IRDA_DEBUG
	lrf_debug_tag(debug_title, "q", false);
	Serial.println((unsigned char)message,DEC);
#endif
	if(message != LRFIRMessage_Unknown)
		lrfIrda.queued = message;
}

void lrf_irda_write_queued_message(void)
{
#if LRF_IRDA_DEBUG
	lrf_debug_tag(debug_title, "qw");
#endif
	lrf_irda_write_message(lrfIrda.queued);
	lrfIrda.queued = LRFIRMessage_Unknown;
}

//------------------------------------------------------------
#pragma mark - Writing
//------------------------------------------------------------

void lrf_irda_write_message(LRFIRMessage message)
{
	unsigned char i = 8;
	unsigned char data = 0x01 | (message << 4) | (message << 1);
	
	// disable interrupts
	noInterrupts();
	
	// disable irda read
	lrf_irda_read_disable();
	
	// do the start bit
	lrf_irda_write_high();
	delay(LRF_IRDA_DELAY_BYTE);
	
	// value should begin with a 0 and end with a 1 (so 32-bits inside)
	do
	{
		i--;
		if((data >> i) & 0x01)	lrf_irda_write_high();
		else					lrf_irda_write_low();
		delay(LRF_IRDA_DELAY_BIT);
	}
	while (i);
	
	// go low again
	lrf_irda_write_low();
	delay(LRF_IRDA_DELAY_BIT);

	// enable irda read (also clears flag - addressing bug)
	lrf_irda_read_enable();
	
	// re-enable interrupts
	interrupts();
}

//------------------------------------------------------------
#pragma mark - Reading
//------------------------------------------------------------

#define lrf_irda_read_bit()		((PIND & LRF_IRDA_IN) > 0)

void lrf_irda_read_message(void)
{
	unsigned char i = 0;	// byte counter
	unsigned char data = 0;
	unsigned int t = 0xfff;
	
	// disable all interrupts
	noInterrupts();
	
	while(lrf_irda_read_bit() == 0 && t) t--;	// hold for rising-edge
	if(t > 0)
	{
		delay(2);
		
		while(i<8)
		{
			data = data << 1;
			
			if(lrf_irda_read_bit() == 0) data += 1;
			delay(LRF_IRDA_DELAY_BIT);
			i++;
		}
		
		// let's trim our edge bits, and compare the middle 3-bit sections
		unsigned char msgA = (data >> 1) & 0b00000111;
		unsigned char msgB = (data >> 4) & 0b00000111;
		
		if(msgA == msgB && msgA != 0 && msgA != 7)
		{
			lrfIrda.hasMessage = true;
			lrfIrda.dispatchedMessage = false;
			lrfIrda.message = (LRFIRMessage)msgA;
			lrf_irda_read_disable();
		}
	}

	// re-enable interrupts
	interrupts();
}




