//
//  LRF.cpp
//  LittleRobotFriends
//
//  Created by Mark Argo on 2014-05-20.
//  Copyright (c) 2014 Aesthetec Studio Inc. All rights reserved.
//

/*
 THIS SOFTWARE IS PROVIDED “AS IS”, WITHOUT ANY REPRESENTATIONS, CONDITIONS, 
 AND/OR WARRANTIES OF ANY KIND.  WITHOUT LIMITATION, AESTHETEC STUDIO AND ITS 
 AFFILIATES, LICENSORS, SUPPLIERS, CONTRIBUTORS, SUBCONTRACTORS, DISTRIBUTORS 
 AND ALL CONTRIBUTORS DISCLAIM ANY EXPRESS OR IMPLIED REPRESENTATIONS, 
 CONDITIONS, OR WARRANTIES OF MERCHANTABILITY, MERCHANTABLE QUALITY, SATISFACTORY 
 QUALITY, NON-INFRINGEMENT, TITLE, DURABILITY, OR FITNESS FOR A PARTICULAR 
 PURPOSE, WHETHER ARISING BY STATUTE, COURSE OF DEALING, USAGE OF TRADE, OR 
 OTHERWISE.  EXCEPT AS OTHERWISE PROVIDED IN THIS AGREEMENT, YOU SHALL BEAR 
 THE ENTIRE RISK FOR ANY USE OR ANY OTHER EXPLOITATION MADE BY YOU OF ANY 
 RIGHTS IN THE COVERED SOFTWARE.
 
 Additional copyright information found at http://littlerobotfriends.com/legal/
*/

#include "LRF.h"
#include "LRFHardware.h"
#include "LRFSpeech.h"
#include "LRFEyes.h"
#include "LRFSensors.h"
#include "LRFEvents.h"
#include "LRFInfrared.h"
#include "LRFPower.h"
#include "LRFExpressions.h"
#include "LRFPersonality.h"
#include "LRFMemory.h"
#include "LRFDebug.h"
#include "LRFGames.h"

const char debug_title[] = "lrf";

LittleRobotFriend lrf;

//-------------------------------------------------------------
#pragma mark - Move these eventually?
//-------------------------------------------------------------

void lrf_print_hardware_version(void)
{
	unsigned char major = lrf_memory_read(LRF_MEMORY_HARDWARE_VERSION_MAJOR_ADDR);
	unsigned char minor = lrf_memory_read(LRF_MEMORY_HARDWARE_VERSION_MINOR_ADDR);
	lrf_debug_tag(debug_title, "hw:", false);
	lrf_debug_pair(major, minor);
}

void lrf_print_firmware_version(void)
{
	unsigned char major = lrf_memory_read(LRF_MEMORY_FIRMWARE_VERSION_MAJOR_ADDR);
	unsigned char minor = lrf_memory_read(LRF_MEMORY_FIRMWARE_VERSION_MINOR_ADDR);
	lrf_debug_tag(debug_title, "fw:", false);
	lrf_debug_pair(major, minor);
}

void lrf_reset_test_memories(void)
{
	lrf_memory_write(LRF_MEMORY_HARDWARE_TEST_ADDR, 0);
	lrf_memory_write(LRF_MEMORY_HAS_READ_SHOELACES_ADDR, 0);
}

void lrf_implant_memories(bool isHoliday)
{
	// write hardware version
	if(isHoliday)
	{
		lrf_memory_write(LRF_MEMORY_HARDWARE_VERSION_MAJOR_ADDR, 0x01);
		lrf_memory_write(LRF_MEMORY_HARDWARE_VERSION_MINOR_ADDR, 0x00);
	}
	else
	{
		lrf_memory_write(LRF_MEMORY_HARDWARE_VERSION_MAJOR_ADDR, 0x01);
		lrf_memory_write(LRF_MEMORY_HARDWARE_VERSION_MINOR_ADDR, 0x01);
	}
	
	// write firmware version
	lrf_memory_write(LRF_MEMORY_FIRMWARE_VERSION_MAJOR_ADDR, LRF_FIRMWARE_MAJOR);
	lrf_memory_write(LRF_MEMORY_FIRMWARE_VERSION_MINOR_ADDR, LRF_FIRMWARE_MINOR);
	
	// write hardware test
	lrf_memory_write(LRF_MEMORY_HARDWARE_TEST_ADDR, LRF_MEMORY_HARDWARE_TEST_VALUE);
	
	// write shoelaces read
	lrf_memory_write(LRF_MEMORY_HAS_READ_SHOELACES_ADDR, LRF_MEMORY_HAS_READ_SHOELACES_VALUE);
	
	// write personality
	lrf_memory_write(LRF_MEMORY_PERSONALITY_START_ADDR, LRF_PERSONALITY_LEVEL_NORMAL);
	lrf_memory_write(LRF_MEMORY_PERSONALITY_START_ADDR+1, LRF_PERSONALITY_LEVEL_NORMAL);
	lrf_memory_write(LRF_MEMORY_PERSONALITY_START_ADDR+2, LRF_PERSONALITY_LEVEL_NORMAL);
	lrf_memory_write(LRF_MEMORY_PERSONALITY_START_ADDR+3, LRF_PERSONALITY_LEVEL_NORMAL);
	lrf_memory_write(LRF_MEMORY_PERSONALITY_START_ADDR+4, LRF_PERSONALITY_LEVEL_NORMAL);
	lrf_memory_write(LRF_MEMORY_PERSONALITY_START_ADDR+5, LRF_PERSONALITY_LEVEL_NORMAL);
	
	// write signature
	lrf_memory_write(LRF_MEMORY_SIGNATURE_ADDR, 1);
}

//-------------------------------------------------------------
#pragma mark - Setup & Loop
//-------------------------------------------------------------

void LittleRobotFriend::setup()
{
	// setup the serial output
	Serial.begin(9600);

	// setup all our libraries
	lrf_hardware_setup();
	
	// holiday hardware test
	unsigned int holiday = (lrf_memory_read(0x08) << 8) | lrf_memory_read(0x09);
	if(holiday == 0xffff)
	{
		lrf_debug_tag(debug_title, "holiday");
		lrf_implant_memories(true);
	}
	
	// firmware version check
	unsigned char major = lrf_memory_read(LRF_MEMORY_FIRMWARE_VERSION_MAJOR_ADDR);
	unsigned char minor = lrf_memory_read(LRF_MEMORY_FIRMWARE_VERSION_MINOR_ADDR);
	if(major != LRF_FIRMWARE_MAJOR || minor != LRF_FIRMWARE_MINOR)
	{
		lrf_memory_write(LRF_MEMORY_FIRMWARE_VERSION_MAJOR_ADDR, LRF_FIRMWARE_MAJOR);
		lrf_memory_write(LRF_MEMORY_FIRMWARE_VERSION_MINOR_ADDR, LRF_FIRMWARE_MINOR);
	}
	
	// print our hardware & firmware versions
	lrf_print_hardware_version();
	lrf_print_firmware_version();
	delay(50);
	
	// test our hardware
	if(LRF_FORCE_HARDWARE_TEST || (lrf_memory_read(LRF_MEMORY_HARDWARE_TEST_ADDR) != LRF_MEMORY_HARDWARE_TEST_VALUE))
	{
#if LRF_ADVANCED_HARDWARE_TEST
		lrf_hardware_test_advanced();
#else
		lrf_hardware_test();
#endif
	}
	
	
	// setup the power functionality
	lrf_power_setup();
	
	// Setup the personality & expressions
	lrf_personality_setup();
	lrf_expressions_setup();
	
	// load saved settings from memory
	lrf_memory_load();
	
	// setup other libraries
	lrf_sensors_setup();
	lrf_events_setup();
	lrf_irda_setup();
	
	// global enable of interrupts
	interrupts();
	
#if LRF_DEBUG_CONSOLE_ENABLED
	lrf_debug_console();
#endif
	
	// Let's do our power-up message
	lrf_events_add(LRFEvent_PowerUp);	
}

void LittleRobotFriend::loop()
{
	// let's sense the world
	lrf_sensors_process();
	
	// did we recognize anything different?
	lrf_events_process();
	
	// if so, then let's react
	lrf_expressions_process();
	
	// if we need to send some ir data, let's do it here
	lrf_irda_process();
	
	// and finally check our power state
	lrf_power_process();
}


//-------------------------------------------------------------
#pragma mark - Expressing
//-------------------------------------------------------------

void LittleRobotFriend::blink(LRFPattern pattern)
{
	lrf_eyes_play_pattern_and_block(pattern,false);
}

void LittleRobotFriend::blink(LRFPattern pattern, unsigned int duration)
{
	lrf_eyes_play_pattern_duration_and_block(pattern,duration,false);
}

void LittleRobotFriend::say(LRFSound sound)
{
	lrf_speech_play_sound_and_block(sound,false);
}

void LittleRobotFriend::blinkAndSay(LRFPattern pattern, LRFSound sound)
{
	lrf_expressions_load_pattern_and_sounds_and_block(pattern, &sound, 1);
}

void LittleRobotFriend::blinkAndSay(LRFPattern pattern, LRFSound *sounds, char soundCount)
{
	lrf_expressions_load_pattern_and_sounds_and_block(pattern, sounds, soundCount);
}

//-------------------------------------------------------------
#pragma mark - Events
//-------------------------------------------------------------

void LittleRobotFriend::setEventHandler(LRFEvent event, LRFEventHandler handler)
{
	lrfEventHandlers[event] = handler;
}

//-------------------------------------------------------------
#pragma mark - Infrared
//-------------------------------------------------------------

LRFIRMessage LittleRobotFriend::readIRMessage(void)
{
	return lrfIrda.message;
}

void LittleRobotFriend::sendIRMessage(LRFIRMessage message)
{
	lrf_irda_write_message(message);
}

//-------------------------------------------------------------
#pragma mark - Sensors
//-------------------------------------------------------------

unsigned char LittleRobotFriend::readTouch()
{
	unsigned char value;
	lrf_touch_read(&value);
	return value;
}

unsigned char LittleRobotFriend::readLight()
{
	unsigned char value;
	lrf_light_read(&value);
	return value;
}

unsigned char LittleRobotFriend::readMicrophone()
{
	unsigned char value;
	lrf_mic_read(&value);
	return value;
}

//-------------------------------------------------------------
#pragma mark - Outputs
//-------------------------------------------------------------

void LittleRobotFriend::setLeftLED(unsigned char red, unsigned char green, unsigned char blue, bool update)
{
	lrfEyes.currentLeft.red = (red < LRF_RGB_VALUE_MAX) ? red : LRF_RGB_VALUE_MAX;
	lrfEyes.currentLeft.green = (green < LRF_RGB_VALUE_MAX) ? green : LRF_RGB_VALUE_MAX;
	lrfEyes.currentLeft.blue = (blue < LRF_RGB_VALUE_MAX) ? blue : LRF_RGB_VALUE_MAX;
	if(update) lrf_eyes_update();
}

void LittleRobotFriend::setRightLED(unsigned char red, unsigned char green, unsigned char blue, bool update)
{
	lrfEyes.currentRight.red = (red < LRF_RGB_VALUE_MAX) ? red : LRF_RGB_VALUE_MAX;
	lrfEyes.currentRight.green = (green < LRF_RGB_VALUE_MAX) ? green : LRF_RGB_VALUE_MAX;
	lrfEyes.currentRight.blue = (blue < LRF_RGB_VALUE_MAX) ? blue : LRF_RGB_VALUE_MAX;
	if(update) lrf_eyes_update();
}

void LittleRobotFriend::setBothLEDs(unsigned char red, unsigned char green, unsigned char blue, bool update)
{
	setRightLED(red, green, blue, false);
	setLeftLED(red, green, blue, false);
	if(update) lrf_eyes_update();
}

void LittleRobotFriend::setSpeaker(unsigned int frequency, unsigned int duration)
{
	lrf_speaker_on();
	lrf_speaker_set_frequency(frequency);
	delay(duration);
	lrf_speaker_off();
}

//-------------------------------------------------------------
#pragma mark - System
//-------------------------------------------------------------

void LittleRobotFriend::setBoredom(unsigned int timeout, unsigned char count)
{
	lrfEvents.boredTimeoutReload = timeout * 1000;
	lrfEvents.boredCountReload = count;
}

void LittleRobotFriend::sleep(void)
{
	lrf_power_sleep();
}









