/// <reference path="../localtypings/vscode-debug-protocol.d.ts" />
/// <reference path="../localtypings/pxtparts.d.ts" />
/// <reference path="../typings/globals/bluebird/index.d.ts" />
declare namespace pxsim {
    interface AllocatorOpts {
        boardDef: BoardDefinition;
        partDefs: Map<PartDefinition>;
        partsList: string[];
        fnArgs: any;
        getBBCoord: (loc: BBLoc) => visuals.Coord;
    }
    interface AllocatorResult {
        partsAndWires: PartAndWiresInst[];
        requiresBreadboard?: boolean;
    }
    interface PartInst {
        name: string;
        simulationBehavior?: string;
        visual: PartVisualDefinition;
        bbFit: PartBBFit;
        startColumnIdx: number;
        startRowIdx: number;
        breadboardConnections: BBLoc[];
        params: Map<string>;
    }
    interface WireInst {
        start: Loc;
        end: Loc;
        color: string;
    }
    interface AssemblyStep {
        part?: boolean;
        wireIndices?: number[];
    }
    interface PartAndWiresInst {
        part?: PartInst;
        wires?: WireInst[];
        assembly: AssemblyStep[];
    }
    interface PartBBFit {
        xOffset: number;
        yOffset: number;
        rowCount: number;
        colCount: number;
    }
    function readPin(arg: string): MicrobitPin;
    function allocateDefinitions(opts: AllocatorOpts): AllocatorResult;
}
/**
 * Heavily adapted from https://github.com/Microsoft/vscode-debugadapter-node
 * and altered to run in a browser and communcate via JSON over a websocket
 * rather than through stdin and stdout
 */
declare namespace pxsim.protocol {
    /**
     * Host for debug session that is responsible with communication with
     * the debugger's user interface.
     */
    interface DebugSessionHost {
        send(msg: string): void;
        onData(cb: (msg: DebugProtocol.ProtocolMessage) => void): void;
        onError(cb: (e?: any) => void): void;
        onClose(cb: () => void): void;
        close(): void;
    }
    class Message implements DebugProtocol.ProtocolMessage {
        seq: number;
        type: string;
        constructor(type: string);
    }
    class Response extends Message implements DebugProtocol.Response {
        request_seq: number;
        success: boolean;
        command: string;
        constructor(request: DebugProtocol.Request, message?: string);
    }
    class Event extends Message implements DebugProtocol.Event {
        event: string;
        constructor(event: string, body?: any);
    }
    class Source implements DebugProtocol.Source {
        name: string;
        path: string;
        sourceReference: number;
        constructor(name: string, path: string, id?: number, origin?: string, data?: any);
    }
    class Scope implements DebugProtocol.Scope {
        name: string;
        variablesReference: number;
        expensive: boolean;
        constructor(name: string, reference: number, expensive?: boolean);
    }
    class StackFrame implements DebugProtocol.StackFrame {
        id: number;
        source: Source;
        line: number;
        column: number;
        name: string;
        constructor(i: number, nm: string, src?: Source, ln?: number, col?: number);
    }
    class Thread implements DebugProtocol.Thread {
        id: number;
        name: string;
        constructor(id: number, name: string);
    }
    class Variable implements DebugProtocol.Variable {
        name: string;
        value: string;
        variablesReference: number;
        constructor(name: string, value: string, ref?: number, indexedVariables?: number, namedVariables?: number);
    }
    class Breakpoint implements DebugProtocol.Breakpoint {
        verified: boolean;
        constructor(verified: boolean, line?: number, column?: number, source?: Source);
    }
    class Module implements DebugProtocol.Module {
        id: number | string;
        name: string;
        constructor(id: number | string, name: string);
    }
    class CompletionItem implements DebugProtocol.CompletionItem {
        label: string;
        start: number;
        length: number;
        constructor(label: string, start: number, length?: number);
    }
    class StoppedEvent extends Event implements DebugProtocol.StoppedEvent {
        body: {
            reason: string;
            threadId: number;
        };
        constructor(reason: string, threadId: number, exception_text?: string);
    }
    class ContinuedEvent extends Event implements DebugProtocol.ContinuedEvent {
        body: {
            threadId: number;
        };
        constructor(threadId: number, allThreadsContinued?: boolean);
    }
    class InitializedEvent extends Event implements DebugProtocol.InitializedEvent {
        constructor();
    }
    class TerminatedEvent extends Event implements DebugProtocol.TerminatedEvent {
        constructor(restart?: boolean);
    }
    class OutputEvent extends Event implements DebugProtocol.OutputEvent {
        body: {
            category: string;
            output: string;
            data?: any;
        };
        constructor(output: string, category?: string, data?: any);
    }
    class ThreadEvent extends Event implements DebugProtocol.ThreadEvent {
        body: {
            reason: string;
            threadId: number;
        };
        constructor(reason: string, threadId: number);
    }
    class BreakpointEvent extends Event implements DebugProtocol.BreakpointEvent {
        body: {
            reason: string;
            breakpoint: Breakpoint;
        };
        constructor(reason: string, breakpoint: Breakpoint);
    }
    class ModuleEvent extends Event implements DebugProtocol.ModuleEvent {
        body: {
            reason: 'new' | 'changed' | 'removed';
            module: Module;
        };
        constructor(reason: 'new' | 'changed' | 'removed', module: Module);
    }
    class ProtocolServer {
        private host;
        private _pendingRequests;
        private _sequence;
        start(host: DebugSessionHost): void;
        stop(): void;
        sendEvent(event: DebugProtocol.Event): void;
        sendResponse(response: DebugProtocol.Response): void;
        sendRequest(command: string, args: any, timeout: number, cb: (response: DebugProtocol.Response) => void): void;
        private send(typ, message);
        protected dispatchRequest(request: DebugProtocol.Request): void;
    }
    class DebugSession extends ProtocolServer {
        private _debuggerLinesStartAt1;
        private _debuggerColumnsStartAt1;
        private _debuggerPathsAreURIs;
        private _clientLinesStartAt1;
        private _clientColumnsStartAt1;
        private _clientPathsAreURIs;
        private _isServer;
        shutdown(): void;
        protected dispatchRequest(request: DebugProtocol.Request): void;
        protected initializeRequest(response: DebugProtocol.InitializeResponse, args: DebugProtocol.InitializeRequestArguments): void;
        protected disconnectRequest(response: DebugProtocol.DisconnectResponse, args: DebugProtocol.DisconnectArguments): void;
        protected launchRequest(response: DebugProtocol.LaunchResponse, args: DebugProtocol.LaunchRequestArguments): void;
        protected attachRequest(response: DebugProtocol.AttachResponse, args: DebugProtocol.AttachRequestArguments): void;
        protected setBreakPointsRequest(response: DebugProtocol.SetBreakpointsResponse, args: DebugProtocol.SetBreakpointsArguments): void;
        protected setFunctionBreakPointsRequest(response: DebugProtocol.SetFunctionBreakpointsResponse, args: DebugProtocol.SetFunctionBreakpointsArguments): void;
        protected setExceptionBreakPointsRequest(response: DebugProtocol.SetExceptionBreakpointsResponse, args: DebugProtocol.SetExceptionBreakpointsArguments): void;
        protected configurationDoneRequest(response: DebugProtocol.ConfigurationDoneResponse, args: DebugProtocol.ConfigurationDoneArguments): void;
        protected continueRequest(response: DebugProtocol.ContinueResponse, args: DebugProtocol.ContinueArguments): void;
        protected nextRequest(response: DebugProtocol.NextResponse, args: DebugProtocol.NextArguments): void;
        protected stepInRequest(response: DebugProtocol.StepInResponse, args: DebugProtocol.StepInArguments): void;
        protected stepOutRequest(response: DebugProtocol.StepOutResponse, args: DebugProtocol.StepOutArguments): void;
        protected stepBackRequest(response: DebugProtocol.StepBackResponse, args: DebugProtocol.StepBackArguments): void;
        protected restartFrameRequest(response: DebugProtocol.RestartFrameResponse, args: DebugProtocol.RestartFrameArguments): void;
        protected gotoRequest(response: DebugProtocol.GotoResponse, args: DebugProtocol.GotoArguments): void;
        protected pauseRequest(response: DebugProtocol.PauseResponse, args: DebugProtocol.PauseArguments): void;
        protected sourceRequest(response: DebugProtocol.SourceResponse, args: DebugProtocol.SourceArguments): void;
        protected threadsRequest(response: DebugProtocol.ThreadsResponse): void;
        protected stackTraceRequest(response: DebugProtocol.StackTraceResponse, args: DebugProtocol.StackTraceArguments): void;
        protected scopesRequest(response: DebugProtocol.ScopesResponse, args: DebugProtocol.ScopesArguments): void;
        protected variablesRequest(response: DebugProtocol.VariablesResponse, args: DebugProtocol.VariablesArguments): void;
        protected setVariableRequest(response: DebugProtocol.SetVariableResponse, args: DebugProtocol.SetVariableArguments): void;
        protected evaluateRequest(response: DebugProtocol.EvaluateResponse, args: DebugProtocol.EvaluateArguments): void;
        protected stepInTargetsRequest(response: DebugProtocol.StepInTargetsResponse, args: DebugProtocol.StepInTargetsArguments): void;
        protected gotoTargetsRequest(response: DebugProtocol.GotoTargetsResponse, args: DebugProtocol.GotoTargetsArguments): void;
        protected completionsRequest(response: DebugProtocol.CompletionsResponse, args: DebugProtocol.CompletionsArguments): void;
        /**
         * Override this hook to implement custom requests.
         */
        protected customRequest(command: string, response: DebugProtocol.Response, args: any): void;
        protected sendErrorResponse(response: DebugProtocol.Response, codeOrMessage: number | DebugProtocol.Message, format?: string, variables?: any): void;
        protected convertClientLineToDebugger(line: number): number;
        protected convertDebuggerLineToClient(line: number): number;
        protected convertClientColumnToDebugger(column: number): number;
        protected convertDebuggerColumnToClient(column: number): number;
        protected convertClientPathToDebugger(clientPath: string): string;
        protected convertDebuggerPathToClient(debuggerPath: string): string;
        private static path2uri(str);
        private static uri2path(url);
        private static _formatPIIRegexp;
        private static formatPII(format, excludePII, args);
    }
}
declare namespace pxsim.util {
    function injectPolyphils(): void;
    class Lazy<T> {
        private _func;
        private _value;
        private _evaluated;
        constructor(_func: () => T);
        value: T;
    }
    function getNormalizedParts(path: string): string[];
    function normalizePath(path: string): string;
    function relativePath(fromDir: string, toFile: string): string;
    function pathJoin(...paths: string[]): string;
}
declare namespace pxsim {
    function getWarningMessage(msg: string): DebuggerWarningMessage;
    class BreakpointMap {
        fileMap: {
            [index: string]: [number, DebugProtocol.Breakpoint][];
        };
        idMap: {
            [index: number]: DebugProtocol.Breakpoint;
        };
        constructor(breakpoints: [number, DebugProtocol.Breakpoint][]);
        getById(id: number): DebugProtocol.Breakpoint;
        verifyBreakpoint(path: string, breakpoint: DebugProtocol.SourceBreakpoint): [number, DebugProtocol.Breakpoint];
    }
    function getBreakpointMsg(s: pxsim.StackFrame, brkId: number): DebuggerBreakpointMessage;
    interface SimLaunchArgs extends DebugProtocol.LaunchRequestArguments {
        projectDir: string;
    }
    class SimDebugSession extends protocol.DebugSession {
        private static THREAD_ID;
        private driver;
        private lastBreak;
        private state;
        private projectDir;
        private breakpoints;
        constructor(container: HTMLElement);
        runCode(js: string, parts: string[], fnArgs: Map<string>, breakpoints: BreakpointMap, board: pxsim.BoardDefinition): void;
        stopSimulator(unload?: boolean): void;
        protected initializeRequest(response: DebugProtocol.InitializeResponse, args: DebugProtocol.InitializeRequestArguments): void;
        protected disconnectRequest(response: DebugProtocol.DisconnectResponse, args: DebugProtocol.DisconnectArguments): void;
        protected launchRequest(response: DebugProtocol.LaunchResponse, args: SimLaunchArgs): void;
        protected setBreakPointsRequest(response: DebugProtocol.SetBreakpointsResponse, args: DebugProtocol.SetBreakpointsArguments): void;
        protected continueRequest(response: DebugProtocol.ContinueResponse, args: DebugProtocol.ContinueArguments): void;
        protected nextRequest(response: DebugProtocol.NextResponse, args: DebugProtocol.NextArguments): void;
        protected stepInRequest(response: DebugProtocol.StepInResponse, args: DebugProtocol.StepInArguments): void;
        protected stepOutRequest(response: DebugProtocol.StepOutResponse, args: DebugProtocol.StepOutArguments): void;
        protected pauseRequest(response: DebugProtocol.PauseResponse, args: DebugProtocol.PauseArguments): void;
        protected threadsRequest(response: DebugProtocol.ThreadsResponse): void;
        protected stackTraceRequest(response: DebugProtocol.StackTraceResponse, args: DebugProtocol.StackTraceArguments): void;
        protected scopesRequest(response: DebugProtocol.ScopesResponse, args: DebugProtocol.ScopesArguments): void;
        protected variablesRequest(response: DebugProtocol.VariablesResponse, args: DebugProtocol.VariablesArguments): void;
        private onDebuggerBreakpoint(breakMsg);
        private onDebuggerWarning(warnMsg);
        private onDebuggerResume();
        private onStateChanged(state);
        private fixBreakpoints();
    }
}
declare namespace pxsim {
    interface SimulatorRunMessage extends SimulatorMessage {
        type: "run";
        id?: string;
        boardDefinition?: BoardDefinition;
        frameCounter?: number;
        options?: any;
        parts?: string[];
        partDefinitions?: Map<PartDefinition>;
        fnArgs?: any;
        code: string;
        mute?: boolean;
        highContrast?: boolean;
        cdnUrl?: string;
    }
    interface SimulatorMuteMessage extends SimulatorMessage {
        type: "mute";
        mute: boolean;
    }
    interface SimulatorDocMessage extends SimulatorMessage {
        docType?: string;
        src?: string;
        localToken?: string;
    }
    interface SimulatorFileLoadedMessage extends SimulatorMessage {
        name: string;
        locale: string;
        content?: string;
    }
    interface SimulatorReadyMessage extends SimulatorMessage {
        type: "ready";
        frameid: string;
    }
    interface SimulatorTopLevelCodeFinishedMessage extends SimulatorMessage {
        type: "toplevelcodefinished";
    }
    interface SimulatorDocsReadyMessage extends SimulatorMessage {
    }
    interface SimulatorStateMessage extends SimulatorMessage {
        frameid?: string;
        runtimeid?: string;
        state: string;
    }
    interface SimulatorEventBusMessage extends SimulatorMessage {
        id: number;
        eventid: number;
        value?: number;
    }
    interface SimulatorSerialMessage extends SimulatorMessage {
        id: string;
        data: string;
        sim?: boolean;
    }
    interface SimulatorCommandMessage extends SimulatorMessage {
        type: "simulator";
        command: "modal" | "restart";
        header?: string;
        body?: string;
        copyable?: string;
        linkButtonHref?: string;
        linkButtonLabel?: string;
    }
    interface SimulatorRadioPacketMessage extends SimulatorMessage {
        type: "radiopacket";
        rssi: number;
        serial: number;
        time: number;
        payload: SimulatorRadioPacketPayload;
    }
    interface SimulatorInfraredPacketMessage extends SimulatorMessage {
        type: "irpacket";
        packet: Uint8Array;
    }
    interface SimulatorRadioPacketPayload {
        type: number;
        groupId: number;
        stringData?: string;
        numberData?: number;
    }
    interface SimulatorCustomMessage extends SimulatorMessage {
        type: "custom";
        content: any;
    }
    interface SimulatorScreenshotMessage extends SimulatorMessage {
        type: "screenshot";
        data: string;
    }
    interface TutorialMessage extends SimulatorMessage {
        type: "tutorial";
        tutorial: string;
        subtype: string;
    }
    interface TutorialStepInfo {
        fullscreen?: boolean;
        hasHint?: boolean;
        content?: string;
        headerContent?: string;
    }
    interface TutorialLoadedMessage extends TutorialMessage {
        subtype: "loaded";
        showCategories?: boolean;
        stepInfo: TutorialStepInfo[];
        toolboxSubset?: {
            [index: string]: number;
        };
    }
    interface TutorialStepChangeMessage extends TutorialMessage {
        subtype: "stepchange";
        step: number;
    }
    namespace Embed {
        function start(): void;
        function stop(): void;
        function run(msg: SimulatorRunMessage): void;
    }
}
declare namespace pxsim.instructions {
    interface RenderPartsOptions {
        name: string;
        boardDef: BoardDefinition;
        parts: string[];
        partDefinitions: Map<PartDefinition>;
        fnArgs?: any;
    }
    function renderParts(options: RenderPartsOptions): void;
}
declare namespace pxsim {
    var quiet: boolean;
    function check(cond: boolean, msg?: string): void;
    function noRefCounting(): void;
    function enableFloatingPoint(): void;
    class RefObject {
        id: number;
        refcnt: number;
        constructor();
        destroy(): void;
        print(): void;
    }
    function noLeakTracking(r: RefObject): void;
    class FnWrapper {
        func: LabelFn;
        caps: any[];
        a0: any;
        a1: any;
        a2: any;
        cb: ResumeFn;
        constructor(func: LabelFn, caps: any[], a0: any, a1: any, a2: any, cb: ResumeFn);
    }
    interface VTable {
        name: string;
        methods: LabelFn[];
        refmask: boolean[];
    }
    class RefRecord extends RefObject {
        fields: any[];
        vtable: VTable;
        destroy(): void;
        isRef(idx: number): boolean;
        print(): void;
    }
    class RefAction extends RefObject {
        fields: any[];
        reflen: number;
        func: LabelFn;
        isRef(idx: number): boolean;
        ldclo(n: number): any;
        destroy(): void;
        print(): void;
    }
    namespace pxtcore {
        function mkAction(reflen: number, len: number, fn: LabelFn): RefAction;
        function runAction3(a: RefAction, a0: any, a1: any, a2: any): void;
        function runAction2(a: RefAction, a0: any, a1: any): void;
        function runAction1(a: RefAction, v: any): void;
        function runAction0(a: RefAction): void;
    }
    class RefLocal extends RefObject {
        v: number;
        print(): void;
    }
    class RefRefLocal extends RefObject {
        v: any;
        destroy(): void;
        print(): void;
    }
    interface MapEntry {
        key: number;
        val: any;
    }
    class RefMap extends RefObject {
        vtable: number;
        data: MapEntry[];
        findIdx(key: number): number;
        destroy(): void;
        print(): void;
    }
    function decr(v: any): void;
    function initString(v: string): string;
    function incr(v: any): any;
    function dumpLivePointers(): void;
    namespace numops {
        function toString(v: any): string;
        function toBoolDecr(v: any): boolean;
        function toBool(v: any): boolean;
    }
    namespace langsupp {
        function toInt(v: number): number;
        function toFloat(v: number): number;
        function ignore(v: any): any;
    }
    namespace pxtcore {
        var incr: typeof pxsim.incr;
        var decr: typeof pxsim.decr;
        function ptrOfLiteral(v: any): any;
        function debugMemLeaks(): void;
        function allocate(): void;
        function templateHash(): number;
        function programHash(): number;
        function programSize(): number;
        function afterProgramPage(): number;
        function toInt(n: number): number;
        function toUInt(n: number): number;
        function toDouble(n: number): number;
        function toFloat(n: number): number;
        function fromInt(n: number): number;
        function fromUInt(n: number): number;
        function fromDouble(n: number): number;
        function fromFloat(n: number): number;
        function fromBool(n: any): boolean;
    }
    namespace pxtrt {
        var incr: typeof pxsim.incr;
        var decr: typeof pxsim.decr;
        function toInt8(v: number): number;
        function toInt16(v: number): number;
        function toInt32(v: number): number;
        function toUInt32(v: number): number;
        function toUInt8(v: number): number;
        function toUInt16(v: number): number;
        function nullFix(v: any): any;
        function nullCheck(v: any): void;
        function panic(code: number): void;
        function stringToBool(s: string): number;
        function ptrToBool(v: any): number;
        function emptyToNull(s: string): any;
        function ldfld(r: RefRecord, idx: number): any;
        function stfld(r: RefRecord, idx: number, v: any): void;
        function ldfldRef(r: RefRecord, idx: number): any;
        function stfldRef(r: RefRecord, idx: number, v: any): void;
        function ldloc(r: RefLocal): number;
        function ldlocRef(r: RefLocal): any;
        function stloc(r: RefLocal, v: any): void;
        function stlocRef(r: RefRefLocal, v: any): void;
        function mkloc(): RefLocal;
        function mklocRef(): RefRefLocal;
        function stclo(a: RefAction, idx: number, v: any): RefAction;
        function runtimeWarning(msg: string): void;
        function mkMap(): RefMap;
        function mapGet(map: RefMap, key: number): any;
        function mapGetRef(map: RefMap, key: number): any;
        function mapSet(map: RefMap, key: number, val: any): void;
        function mapSetRef(map: RefMap, key: number, val: any): void;
        function switch_eq(a: any, b: any): boolean;
        var getGlobalsPtr: any;
    }
    namespace pxtcore {
        function mkClassInstance(vtable: VTable): RefRecord;
        var getNumGlobals: any;
        var RefRecord_destroy: any;
        var RefRecord_print: any;
        var anyPrint: any;
        var dumpDmesg: any;
        var getVTable: any;
        var switch_eq: any;
        var valType: any;
        var typeOf: any;
        var lookupPin: any;
    }
    namespace thread {
        var panic: typeof pxtrt.panic;
        function pause(ms: number): void;
        function runInBackground(a: RefAction): void;
        function forever(a: RefAction): void;
    }
}
declare namespace pxsim {
    class RefCollection extends RefObject {
        private data;
        constructor();
        destroy(): void;
        isValidIndex(x: number): boolean;
        push(x: any): void;
        pop(): any;
        getLength(): number;
        setLength(x: number): void;
        getAt(x: number): any;
        setAt(x: number, y: any): void;
        insertAt(x: number, y: number): void;
        removeAt(x: number): any;
        indexOf(x: number, start: number): number;
        print(): void;
    }
    namespace Array_ {
        function mk(): RefCollection;
        function length(c: RefCollection): number;
        function setLength(c: RefCollection, x: number): void;
        function push(c: RefCollection, x: any): void;
        function pop(c: RefCollection, x: any): any;
        function getAt(c: RefCollection, x: number): any;
        function removeAt(c: RefCollection, x: number): any;
        function insertAt(c: RefCollection, x: number, y: number): void;
        function setAt(c: RefCollection, x: number, y: any): void;
        function indexOf(c: RefCollection, x: any, start: number): number;
        function removeElement(c: RefCollection, x: any): number;
    }
    namespace Math_ {
        function imul(x: number, y: number): number;
        function idiv(x: number, y: number): number;
        function round(n: number): number;
        function ceil(n: number): number;
        function floor(n: number): number;
        function sqrt(n: number): number;
        function pow(x: number, y: number): number;
        function log(n: number): number;
        function exp(n: number): number;
        function sin(n: number): number;
        function cos(n: number): number;
        function tan(n: number): number;
        function asin(n: number): number;
        function acos(n: number): number;
        function atan(n: number): number;
        function atan2(y: number, x: number): number;
        function trunc(x: number): number;
        function random(): number;
        function randomRange(min: number, max: number): number;
    }
    namespace Number_ {
        function lt(x: number, y: number): boolean;
        function le(x: number, y: number): boolean;
        function neq(x: number, y: number): boolean;
        function eq(x: number, y: number): boolean;
        function gt(x: number, y: number): boolean;
        function ge(x: number, y: number): boolean;
        function div(x: number, y: number): number;
        function mod(x: number, y: number): number;
        function toString(x: number): string;
    }
    namespace thumb {
        function adds(x: number, y: number): number;
        function subs(x: number, y: number): number;
        function divs(x: number, y: number): number;
        function muls(x: number, y: number): number;
        function ands(x: number, y: number): number;
        function orrs(x: number, y: number): number;
        function eors(x: number, y: number): number;
        function lsls(x: number, y: number): number;
        function lsrs(x: number, y: number): number;
        function asrs(x: number, y: number): number;
        function ignore(v: any): any;
    }
    namespace String_ {
        function mkEmpty(): string;
        function fromCharCode(code: number): string;
        function toNumber(s: string): number;
        function concat(a: string, b: string): string;
        function substring(s: string, i: number, j: number): string;
        function equals(s1: string, s2: string): boolean;
        function compare(s1: string, s2: string): number;
        function length(s: string): number;
        function isEmpty(s: string): boolean;
        function substr(s: string, start: number, length?: number): string;
        function charAt(s: string, i: number): string;
        function charCodeAt(s: string, i: number): number;
    }
    namespace Boolean_ {
        function toString(v: boolean): string;
        function bang(v: boolean): boolean;
    }
    class RefBuffer extends RefObject {
        data: Uint8Array;
        constructor(data: Uint8Array);
        print(): void;
    }
    namespace BufferMethods {
        enum NumberFormat {
            Int8LE = 1,
            UInt8LE = 2,
            Int16LE = 3,
            UInt16LE = 4,
            Int32LE = 5,
            Int8BE = 6,
            UInt8BE = 7,
            Int16BE = 8,
            UInt16BE = 9,
            Int32BE = 10,
            UInt32LE = 11,
            UInt32BE = 12,
            Float32LE = 13,
            Float64LE = 14,
            Float32BE = 15,
            Float64BE = 16,
        }
        function getNumber(buf: RefBuffer, fmt: NumberFormat, offset: number): number;
        function setNumber(buf: RefBuffer, fmt: NumberFormat, offset: number, r: number): void;
        function createBuffer(size: number): RefBuffer;
        function createBufferFromHex(hex: string): RefBuffer;
        function getBytes(buf: RefBuffer): Uint8Array;
        function getByte(buf: RefBuffer, off: number): number;
        function setByte(buf: RefBuffer, off: number, v: number): void;
        function length(buf: RefBuffer): number;
        function fill(buf: RefBuffer, value: number, offset?: number, length?: number): void;
        function slice(buf: RefBuffer, offset: number, length: number): RefBuffer;
        function shift(buf: RefBuffer, offset: number, start: number, len: number): void;
        function rotate(buf: RefBuffer, offset: number, start: number, len: number): void;
        function write(buf: RefBuffer, dstOffset: number, src: RefBuffer, srcOffset?: number, length?: number): void;
    }
}
declare namespace pxsim.logs {
    interface ILogProps {
        maxEntries?: number;
        maxLineLength?: number;
        maxAccValues?: number;
        onClick?: (entries: ILogEntry[]) => void;
        onTrendChartChanged?: () => void;
        onTrendChartClick?: (entry: ILogEntry) => void;
        chromeExtension?: string;
        useHF2?: boolean;
        vendorId?: string;
        productId?: string;
        nameFilter?: string;
    }
    interface ILogEntry {
        id: number;
        theme: string;
        variable?: string;
        accvalues?: {
            t: number;
            v: number;
        }[];
        time: number;
        value: string;
        source: string;
        count: number;
    }
    class LogViewElement {
        props: ILogProps;
        static counter: number;
        private shouldScroll;
        private entries;
        private serialBuffers;
        private dropSim;
        element: HTMLDivElement;
        private labelElement;
        constructor(props: ILogProps);
        setLabel(text: string, theme?: string): void;
        hasTrends(): boolean;
        rows(): ILogEntry[];
        streamPayload(startTime: number): {
            fields: string[];
            values: number[][];
        };
        registerChromeSerial(): void;
        registerEvents(): void;
        appendEntry(source: string, value: string, theme: string): void;
        renderFiberId: number;
        private scheduleRender(e);
        clear(): void;
        private render();
    }
    function entriesToCSV(entries: ILogEntry[]): string;
    function entryToCSV(entry: ILogEntry): string;
}
declare namespace pxsim {
    namespace U {
        function addClass(el: HTMLElement, cls: string): void;
        function removeClass(el: HTMLElement, cls: string): void;
        function assert(cond: boolean, msg?: string): void;
        function repeatMap<T>(n: number, fn: (index: number) => T): T[];
        function userError(msg: string): Error;
        function now(): number;
        function nextTick(f: () => void): void;
    }
    interface Map<T> {
        [index: string]: T;
    }
    type LabelFn = (s: StackFrame) => StackFrame;
    type ResumeFn = (v?: any) => void;
    interface StackFrame {
        fn: LabelFn;
        pc: number;
        overwrittenPC?: boolean;
        depth: number;
        r0?: any;
        parent: StackFrame;
        retval?: any;
        lambdaArgs?: any[];
        caps?: any[];
        finalCallback?: ResumeFn;
        lastBrkId?: number;
    }
    let runtime: Runtime;
    function getResume(): (v?: any) => void;
    class BaseBoard {
        runOptions: SimulatorRunMessage;
        updateView(): void;
        receiveMessage(msg: SimulatorMessage): void;
        initAsync(msg: SimulatorRunMessage): Promise<void>;
        kill(): void;
        protected serialOutBuffer: string;
        writeSerial(s: string): void;
    }
    class CoreBoard extends BaseBoard {
        id: string;
        bus: pxsim.EventBus;
        updateSubscribers: (() => void)[];
        builtinParts: Map<any>;
        builtinVisuals: Map<() => visuals.IBoardPart<any>>;
        builtinPartVisuals: Map<(xy: visuals.Coord) => visuals.SVGElAndSize>;
        constructor();
        kill(): void;
    }
    function initBareRuntime(): void;
    class EventQueue<T> {
        runtime: Runtime;
        max: number;
        events: T[];
        private awaiters;
        private mHandler;
        private lock;
        constructor(runtime: Runtime);
        push(e: T, notifyOne: boolean): void;
        private poke();
        handler: RefAction;
        addAwaiter(awaiter: (v?: any) => void): void;
    }
    let initCurrentRuntime: () => void;
    let handleCustomMessage: (message: pxsim.SimulatorCustomMessage) => void;
    class Runtime {
        board: BaseBoard;
        numGlobals: number;
        errorHandler: (e: any) => void;
        postError: (e: any) => void;
        stateChanged: () => void;
        dead: boolean;
        running: boolean;
        startTime: number;
        id: string;
        globals: any;
        currFrame: StackFrame;
        entry: LabelFn;
        overwriteResume: (retPC: number) => void;
        getResume: () => ResumeFn;
        run: (cb: ResumeFn) => void;
        setupTop: (cb: ResumeFn) => StackFrame;
        handleDebuggerMsg: (msg: DebuggerMessage) => void;
        runningTime(): number;
        runFiberAsync(a: RefAction, arg0?: any, arg1?: any, arg2?: any): Promise<any>;
        static messagePosted: (data: SimulatorMessage) => void;
        static postMessage(data: SimulatorMessage): void;
        kill(): void;
        updateDisplay(): void;
        private numDisplayUpdates;
        queueDisplayUpdate(): void;
        maybeUpdateDisplay(): void;
        setRunning(r: boolean): void;
        constructor(code: string);
    }
}
declare namespace pxsim {
    interface SimulatorDriverOptions {
        revealElement?: (el: HTMLElement) => void;
        removeElement?: (el: HTMLElement, onComplete?: () => void) => void;
        unhideElement?: (el: HTMLElement) => void;
        onDebuggerWarning?: (wrn: DebuggerWarningMessage) => void;
        onDebuggerBreakpoint?: (brk: DebuggerBreakpointMessage) => void;
        onTraceMessage?: (msg: TraceMessage) => void;
        onDebuggerResume?: () => void;
        onStateChanged?: (state: SimulatorState) => void;
        onSimulatorCommand?: (msg: pxsim.SimulatorCommandMessage) => void;
        onTopLevelCodeEnd?: () => void;
        simUrl?: string;
    }
    enum SimulatorState {
        Unloaded = 0,
        Stopped = 1,
        Running = 2,
        Paused = 3,
    }
    enum SimulatorDebuggerCommand {
        StepInto = 0,
        StepOver = 1,
        StepOut = 2,
        Resume = 3,
        Pause = 4,
    }
    interface SimulatorRunOptions {
        debug?: boolean;
        boardDefinition?: pxsim.BoardDefinition;
        parts?: string[];
        fnArgs?: any;
        aspectRatio?: number;
        partDefinitions?: pxsim.Map<PartDefinition>;
        mute?: boolean;
        highContrast?: boolean;
        cdnUrl?: string;
    }
    interface HwDebugger {
        postMessage: (msg: pxsim.SimulatorMessage) => void;
    }
    class SimulatorDriver {
        container: HTMLElement;
        options: SimulatorDriverOptions;
        private themes;
        private runId;
        private nextFrameId;
        private frameCounter;
        private currentRuntime;
        private listener;
        private traceInterval;
        runOptions: SimulatorRunOptions;
        state: SimulatorState;
        hwdbg: HwDebugger;
        constructor(container: HTMLElement, options?: SimulatorDriverOptions);
        setHwDebugger(hw: HwDebugger): void;
        handleHwDebuggerMsg(msg: pxsim.SimulatorMessage): void;
        setThemes(themes: string[]): void;
        private setState(state);
        postMessage(msg: pxsim.SimulatorMessage, source?: Window): void;
        private createFrame();
        stop(unload?: boolean): void;
        private unload();
        mute(mute: boolean): void;
        private frameCleanupTimeout;
        private cancelFrameCleanup();
        private scheduleFrameCleanup();
        private applyAspectRatio();
        private cleanupFrames();
        hide(completeHandler?: () => void): void;
        unhide(): void;
        run(js: string, opts?: SimulatorRunOptions): void;
        private startFrame(frame);
        private removeEventListeners();
        private handleMessage(msg, source?);
        private addEventListeners();
        resume(c: SimulatorDebuggerCommand): void;
        setBreakpoints(breakPoints: number[]): void;
        setTraceInterval(intervalMs: number): void;
        private handleSimulatorCommand(msg);
        private handleDebuggerMessage(msg);
        private postDebuggerMessage(subtype, data?);
        private nextId();
    }
}
declare namespace pxsim {
    type BoardPin = string;
    interface BBLoc {
        type: "breadboard";
        row: string;
        col: string;
        xOffset?: number;
        yOffset?: number;
        style?: PinStyle;
    }
    interface BoardLoc {
        type: "dalboard";
        pin: BoardPin;
    }
    type Loc = BBLoc | BoardLoc;
    function mkRange(a: number, b: number): number[];
    function parseQueryString(): (key: string) => string;
    class EventBus {
        private runtime;
        private queues;
        private notifyID;
        private notifyOneID;
        nextNotifyEvent: number;
        setNotify(notifyID: number, notifyOneID: number): void;
        constructor(runtime: Runtime);
        private start(id, evid, create);
        listen(id: number | string, evid: number | string, handler: RefAction): void;
        queue(id: number | string, evid: number | string, value?: number): void;
        wait(id: number | string, evid: number | string, cb: (v?: any) => void): void;
    }
    interface AnimationOptions {
        interval: number;
        frame: () => boolean;
        whenDone?: (cancelled: boolean) => void;
        setTimeoutHandle?: number;
    }
    class AnimationQueue {
        private runtime;
        private queue;
        private process;
        constructor(runtime: Runtime);
        cancelAll(): void;
        cancelCurrent(): void;
        enqueue(anim: AnimationOptions): void;
        executeAsync(anim: AnimationOptions): Promise<boolean>;
    }
    namespace AudioContextManager {
        function mute(mute: boolean): void;
        function stop(): void;
        function frequency(): number;
        function tone(frequency: number, gain: number): void;
    }
    interface IPointerEvents {
        up: string;
        down: string;
        move: string;
        leave: string;
    }
    const pointerEvents: {
        up: string;
        down: string;
        move: string;
        leave: string;
    };
}
declare namespace pxsim.visuals {
    interface IBoardPart<T> {
        style: string;
        element: SVGElement;
        overElement?: SVGElement;
        defs: SVGElement[];
        init(bus: EventBus, state: T, svgEl: SVGSVGElement, otherParams: Map<string>): void;
        moveToCoord(xy: visuals.Coord): void;
        updateState(): void;
        updateTheme(): void;
    }
    function translateEl(el: SVGElement, xy: [number, number]): void;
    interface ComposeOpts {
        el1: SVGAndSize<SVGSVGElement>;
        scaleUnit1: number;
        el2: SVGAndSize<SVGSVGElement>;
        scaleUnit2: number;
        margin: [number, number, number, number];
        middleMargin: number;
        maxWidth?: string;
        maxHeight?: string;
    }
    interface ComposeResult {
        host: SVGSVGElement;
        scaleUnit: number;
        under: SVGGElement;
        over: SVGGElement;
        edges: number[];
        toHostCoord1: (xy: Coord) => Coord;
        toHostCoord2: (xy: Coord) => Coord;
    }
    function composeSVG(opts: ComposeOpts): ComposeResult;
    function mkScaleFn(originUnit: number, targetUnit: number): (n: number) => number;
    interface MkImageOpts {
        image: string;
        width: number;
        height: number;
        imageUnitDist: number;
        targetUnitDist: number;
    }
    function mkImageSVG(opts: MkImageOpts): SVGAndSize<SVGImageElement>;
    type Coord = [number, number];
    function findDistSqrd(a: Coord, b: Coord): number;
    function findClosestCoordIdx(a: Coord, bs: Coord[]): number;
    function mkTxt(cx: number, cy: number, size: number, rot: number, txt: string, txtXOffFactor?: number, txtYOffFactor?: number): SVGTextElement;
    type WireColor = "black" | "white" | "gray" | "purple" | "blue" | "green" | "yellow" | "orange" | "red" | "brown" | "pink";
    const GPIO_WIRE_COLORS: string[];
    const WIRE_COLOR_MAP: Map<string>;
    function mapWireColor(clr: WireColor | string): string;
    interface SVGAndSize<T extends SVGElement> {
        el: T;
        y: number;
        x: number;
        w: number;
        h: number;
    }
    type SVGElAndSize = SVGAndSize<SVGElement>;
    const PIN_DIST: number;
    interface BoardView {
        getView(): SVGAndSize<SVGSVGElement>;
        getCoord(pinNm: string): Coord;
        getPinDist(): number;
        highlightPin(pinNm: string): void;
    }
    function rgbToHsl(rgb: [number, number, number]): [number, number, number];
}
declare namespace pxsim.svg {
    function parseString(xml: string): SVGSVGElement;
    function toDataUri(xml: string): string;
    function cursorPoint(pt: SVGPoint, svg: SVGSVGElement, evt: MouseEvent): SVGPoint;
    function rotateElement(el: SVGElement, originX: number, originY: number, degrees: number): void;
    function hasClass(el: SVGElement, cls: string): boolean;
    function addClass(el: SVGElement, cls: string): void;
    function removeClass(el: SVGElement, cls: string): void;
    function hydrate(el: SVGElement, props: any): void;
    function elt(name: string, props?: any): SVGElement;
    function child(parent: Element, name: string, props?: any): SVGElement;
    function mkPath(cls: string, data: string, title?: string): SVGPathElement;
    function path(parent: Element, cls: string, data: string, title?: string): SVGPathElement;
    function fill(el: SVGElement, c: string): void;
    function filter(el: SVGElement, c: string): void;
    function fills(els: SVGElement[], c: string): void;
    function isTouchEnabled(): boolean;
    const touchEvents: {
        "mousedown": string[];
        "mouseup": string[];
        "mousemove": string[];
        "mouseleave": string[];
    };
    function onClick(el: Element, click: (ev: MouseEvent) => void): void;
    function buttonEvents(el: Element, move?: (ev: MouseEvent) => void, start?: (ev: MouseEvent) => void, stop?: (ev: MouseEvent) => void): void;
    function mkLinearGradient(id: string): SVGLinearGradientElement;
    function linearGradient(defs: SVGDefsElement, id: string): SVGLinearGradientElement;
    function setGradientColors(lg: SVGLinearGradientElement, start: string, end: string): void;
    function setGradientValue(lg: SVGLinearGradientElement, percent: string): void;
    function animate(el: SVGElement, cls: string): void;
    function mkTitle(txt: string): SVGTitleElement;
    function title(el: SVGElement, txt: string): SVGTitleElement;
    function toHtmlColor(c: number): string;
}
declare namespace pxsim {
    class Button {
        id: number;
        constructor(id: number);
        pressed: boolean;
        virtual: boolean;
    }
    interface ButtonPairProps {
        ID_BUTTON_A: number;
        ID_BUTTON_B: number;
        ID_BUTTON_AB: number;
        BUTTON_EVT_UP: number;
        BUTTON_EVT_CLICK: number;
    }
    class ButtonPairState {
        props: ButtonPairProps;
        usesButtonAB: boolean;
        aBtn: Button;
        bBtn: Button;
        abBtn: Button;
        constructor(props: ButtonPairProps);
    }
}
declare namespace pxsim {
    class CompassState {
        usesHeading: boolean;
        heading: number;
    }
}
declare namespace pxsim {
    enum PinFlags {
        Unused = 0,
        Digital = 1,
        Analog = 2,
        Input = 4,
        Output = 8,
        Touch = 16,
    }
    class Pin {
        id: number;
        constructor(id: number);
        touched: boolean;
        value: number;
        period: number;
        servoAngle: number;
        mode: PinFlags;
        pitch: boolean;
        pull: number;
        digitalReadPin(): number;
        digitalWritePin(value: number): void;
        setPull(pull: number): void;
        analogReadPin(): number;
        analogWritePin(value: number): void;
        analogSetPeriod(micros: number): void;
        servoWritePin(value: number): void;
        servoSetPulse(pinId: number, micros: number): void;
        isTouched(): boolean;
    }
    interface EdgeConnectorProps {
        pins: number[];
        servos?: {
            [name: string]: number;
        };
    }
    class EdgeConnectorState {
        props: EdgeConnectorProps;
        pins: Pin[];
        constructor(props: EdgeConnectorProps);
        getPin(id: number): Pin;
    }
}
declare namespace pxsim {
    class FileSystemState {
        files: Map<string>;
        append(file: string, content: string): void;
        remove(file: string): void;
    }
}
declare namespace pxsim {
    class LightSensorState {
        usesLightLevel: boolean;
        lightLevel: number;
    }
}
declare namespace pxsim {
    enum NeoPixelMode {
        RGB = 0,
        RGBW = 1,
    }
    type RGBW = [number, number, number, number];
    class NeoPixelState {
        private buffers;
        private colors;
        private dirty;
        updateBuffer(buffer: Uint8Array, pin: number): void;
        getColors(pin: number, mode: NeoPixelMode): RGBW[];
        private readNeoPixelBuffer(inBuffer, outColors, mode);
    }
}
declare namespace pxsim {
    class ToggleState {
        private pin;
        on: boolean;
        constructor(pin: Pin);
        toggle(): void;
    }
    interface ToggleStateConstructor {
        (pin: Pin): ToggleState;
    }
}
declare namespace pxsim.visuals {
    interface BoardViewOptions {
        visual: string | BoardImageDefinition;
        wireframe?: boolean;
        highContrast?: boolean;
    }
    interface BoardHostOpts {
        state: CoreBoard;
        boardDef: BoardDefinition;
        partsList: string[];
        partDefs: Map<PartDefinition>;
        fnArgs: any;
        forceBreadboardLayout?: boolean;
        forceBreadboardRender?: boolean;
        maxWidth?: string;
        maxHeight?: string;
        wireframe?: boolean;
        highContrast?: boolean;
    }
    var mkBoardView: (opts: BoardViewOptions) => BoardView;
    class BoardHost {
        private parts;
        private wireFactory;
        private breadboard;
        private fromBBCoord;
        private fromMBCoord;
        private boardView;
        private view;
        private partGroup;
        private partOverGroup;
        private style;
        private defs;
        private state;
        private useCrocClips;
        constructor(view: BoardView, opts: BoardHostOpts);
        highlightBoardPin(pinNm: string): void;
        highlightBreadboardPin(rowCol: BBLoc): void;
        highlightWire(wire: Wire): void;
        getView(): SVGElement;
        private updateState();
        private getBBCoord(rowCol);
        private getPinCoord(pin);
        getLocCoord(loc: Loc): Coord;
        addPart(partInst: PartInst): IBoardPart<any>;
        addWire(inst: WireInst): Wire;
        addAll(allocRes: AllocatorResult): void;
    }
}
declare namespace pxsim.visuals {
    const BREADBOARD_MID_ROWS: number;
    const BREADBOARD_MID_COLS: number;
    function getColumnName(colIdx: number): string;
    function getRowName(rowIdx: number): string;
    interface GridPin {
        el: SVGElement;
        hoverEl: SVGElement;
        cx: number;
        cy: number;
        row: string;
        col: string;
        group?: string;
    }
    interface GridOptions {
        xOffset?: number;
        yOffset?: number;
        rowCount: number;
        colCount: number;
        rowStartIdx?: number;
        colStartIdx?: number;
        pinDist: number;
        mkPin: () => SVGElAndSize;
        mkHoverPin: () => SVGElAndSize;
        getRowName: (rowIdx: number) => string;
        getColName: (colIdx: number) => string;
        getGroupName?: (rowIdx: number, colIdx: number) => string;
        rowIdxsWithGap?: number[];
        colIdxsWithGap?: number[];
    }
    interface GridResult {
        g: SVGGElement;
        allPins: GridPin[];
    }
    function mkGrid(opts: GridOptions): GridResult;
    interface GridLabel {
        el: SVGTextElement;
        hoverEl: SVGTextElement;
        txt: string;
        group?: string;
    }
    interface BreadboardOpts {
        wireframe?: boolean;
    }
    class Breadboard {
        bb: SVGSVGElement;
        private styleEl;
        private defs;
        private allPins;
        private allLabels;
        private allPowerBars;
        private rowColToPin;
        private rowColToLbls;
        constructor(opts: BreadboardOpts);
        hide(): void;
        updateLocation(x: number, y: number): void;
        getPin(row: string, col: string): GridPin;
        getCoord(rowCol: BBLoc): Coord;
        getPinDist(): number;
        private buildDom();
        getSVGAndSize(): SVGAndSize<SVGSVGElement>;
        highlightLoc(rowCol: BBLoc): void;
    }
}
declare namespace pxsim.visuals {
    function mkBtnSvg(xy: Coord): SVGAndSize<SVGGElement>;
    const BUTTON_PAIR_STYLE: string;
    class ButtonPairView implements IBoardPart<ButtonPairState> {
        element: SVGElement;
        defs: SVGElement[];
        style: string;
        private state;
        private bus;
        private aBtn;
        private bBtn;
        private abBtn;
        init(bus: EventBus, state: ButtonPairState): void;
        moveToCoord(xy: Coord): void;
        updateState(): void;
        updateTheme(): void;
        private mkBtns();
        private attachEvents();
    }
}
declare namespace pxsim.visuals {
    const BOARD_SYTLE: string;
    interface GenericBoardProps {
        visualDef: BoardImageDefinition;
        wireframe?: boolean;
    }
    class GenericBoardSvg implements BoardView {
        props: GenericBoardProps;
        private element;
        private style;
        private defs;
        private g;
        private background;
        private width;
        private height;
        private id;
        private allPins;
        private allLabels;
        private pinNmToLbl;
        private pinNmToPin;
        constructor(props: GenericBoardProps);
        getCoord(pinNm: string): Coord;
        private mkGrayCover(x, y, w, h);
        getView(): SVGAndSize<SVGSVGElement>;
        getPinDist(): number;
        highlightPin(pinNm: string): void;
    }
}
declare namespace pxsim.visuals {
    function mkGenericPartSVG(partVisual: PartVisualDefinition): SVGAndSize<SVGImageElement>;
    class GenericPart implements IBoardPart<any> {
        style: string;
        element: SVGElement;
        defs: SVGElement[];
        constructor(partVisual: PartVisualDefinition);
        moveToCoord(xy: Coord): void;
        init(bus: EventBus, state: any, svgEl: SVGSVGElement): void;
        updateState(): void;
        updateTheme(): void;
    }
}
declare namespace pxsim.visuals {
    function mkMicroServoPart(xy?: Coord): SVGElAndSize;
    class MicroServoView implements IBoardPart<EdgeConnectorState> {
        style: string;
        overElement: SVGElement;
        element: SVGElement;
        defs: SVGElement[];
        state: EdgeConnectorState;
        bus: EventBus;
        private currentAngle;
        private targetAngle;
        private lastAngleTime;
        private pin;
        private crankEl;
        private crankTransform;
        init(bus: EventBus, state: EdgeConnectorState, svgEl: SVGSVGElement, otherParams: Map<string>): void;
        initDom(): void;
        moveToCoord(xy: visuals.Coord): void;
        updateState(): void;
        updateTheme(): void;
    }
}
declare namespace pxsim.visuals {
    const WIRES_CSS: string;
    interface Wire {
        endG: SVGGElement;
        end1: SVGElement;
        end2: SVGElement;
        wires: SVGElement[];
    }
    enum WireEndStyle {
        BBJumper = 0,
        OpenJumper = 1,
        Croc = 2,
    }
    interface WireOpts {
        color?: string;
        colorClass?: string;
        bendFactor?: number;
    }
    function mkWirePart(cp: [number, number], clr: string, croc?: boolean): visuals.SVGAndSize<SVGGElement>;
    class WireFactory {
        private underboard;
        private overboard;
        private boardEdges;
        private getLocCoord;
        styleEl: SVGStyleElement;
        constructor(underboard: SVGGElement, overboard: SVGGElement, boardEdges: number[], styleEl: SVGStyleElement, getLocCoord: (loc: Loc) => Coord);
        private indexOfMin(vs);
        private closestEdgeIdx(p);
        private closestEdge(p);
        private nextWireId;
        private drawWire(pin1, pin2, color);
        private drawWireWithCrocs(pin1, pin2, color, smallPin?);
        addWire(start: Loc, end: Loc, color: string, withCrocs?: boolean): Wire;
    }
}
