#ifndef __LRF_DEBUG_H
#define __LRF_DEBUG_H

#include <Arduino.h>
#include "lrf_constants.h"
#include "system/types/lrf_types.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef enum
{
	LRFDebugLevel_Off = 0,
	LRFDebugLevel_Normal = 1,
	LRFDebugLevel_Verbose = 2,
	LRFDebugLevel_All = 3
} LRFDebugLevel;


#if LRF_DEBUG_OUTPUT

#define LRF_DEBUG_INIT				SerialUSB.begin(115200)
#define LRF_DEBUG_WAIT_FOR_SPACE	(!SerialUSB.available() && SerialUSB.read() != ' ')
#define LRF_DEBUG_WAIT_CLEAR		while(SerialUSB.available()){ SerialUSB.read(); }

// Print a debug string
#define LRF_DEBUG_S(s)			SerialUSB.print(s)
#define LRF_DEBUG_ST(s,t)		SerialUSB.print(s,t)
#define LRF_DEBUG_SNL(s)		SerialUSB.println(s)
#define LRF_DEBUG_STNL(s,t)	SerialUSB.println(s,t)

#else

#define LRF_DEBUG_INIT
#define LRF_DEBUG_WAIT_FOR_SPACE
#define LRF_DEBUG_WAIT_CLEAR

#define LRF_DEBUG_S(s)
#define LRF_DEBUG_ST(s,t)
#define LRF_DEBUG_SNL(s)
#define LRF_DEBUG_STNL(s,t)

#endif

// Various symbols
#define LRF_DEBUG_LINE		LRF_DEBUG_S("------------------------")
#define LRF_DEBUG_NL			LRF_DEBUG_SNL()
#define LRF_DEBUG_HL			{ LRF_DEBUG_LINE; LRF_DEBUG_SNL(); }
#define LRF_DEBUG_SEP			LRF_DEBUG_S(": ")
#define LRF_DEBUG_TAB			LRF_DEBUG_S("\t")
#define LRF_DEBUG_COMMA		LRF_DEBUG_S(", ")
#define LRF_DEBUG_SPACE		LRF_DEBUG_S(" ")
#define LRF_DEBUG_UNDER		LRF_DEBUG_S("_")
#define LRF_DEBUG_OPEN			LRF_DEBUG_S("[ ")
#define LRF_DEBUG_CLOSE		LRF_DEBUG_S(" ]")

// Tagging functions
#define LRF_DEBUG_TAG(l,f)	{ LRF_DEBUG_LINE; LRF_DEBUG_OPEN; LRF_DEBUG_S(l); LRF_DEBUG_UNDER; LRF_DEBUG_S(f); LRF_DEBUG_CLOSE; LRF_DEBUG_NL; }
#define LRF_DEBUG_TAG1(l,f,a)	{ LRF_DEBUG_LINE; LRF_DEBUG_OPEN; LRF_DEBUG_S(l); LRF_DEBUG_UNDER; LRF_DEBUG_S(f); LRF_DEBUG_SEP; LRF_DEBUG_S(a); LRF_DEBUG_CLOSE; LRF_DEBUG_NL; }
#define LRF_DEBUG_TAG1T(l,f,a,t)	{ LRF_DEBUG_LINE; LRF_DEBUG_OPEN; LRF_DEBUG_S(l); LRF_DEBUG_UNDER; LRF_DEBUG_S(f); LRF_DEBUG_SEP; LRF_DEBUG_ST(a,t); LRF_DEBUG_CLOSE; LRF_DEBUG_NL; }
#define LRF_DEBUG_ERROR(s)	{ LRF_DEBUG_S("ERROR: "); LRF_DEBUG_SNL(s); }
#define LRF_DEBUG_SUCCESS(s)	{ LRF_DEBUG_S("SUCCESS: "); LRF_DEBUG_SNL(s); }
#define LRF_DEBUG_RESPONSE(s)	{ LRF_DEBUG_S("RESPONSE: "); LRF_DEBUG_OPEN; LRF_DEBUG_S(s); LRF_DEBUG_CLOSE; LRF_DEBUG_NL; }
#define LRF_DEBUG_INFO(s)			{ LRF_DEBUG_OPEN; LRF_DEBUG_S(s); LRF_DEBUG_CLOSE; }

#define LRF_DEBUG_FUNCTION(s)	{ LRF_DEBUG_S("{"); LRF_DEBUG_S(s); LRF_DEBUG_SNL("}"); }
#define LRF_DEBUG_CLASS(s)	{ LRF_DEBUG_HL; LRF_DEBUG_S("<"); LRF_DEBUG_S(s); LRF_DEBUG_SNL(">"); }
#define LRF_DEBUG_UPDATE(s)	{ LRF_DEBUG_S("..."); LRF_DEBUG_SNL(s); }
#define LRF_DEBUG_DONE	{ LRF_DEBUG_SNL("<done>"); }

// Print a value (with string identifier)
#define LRF_DEBUG_V(s,v)		{ LRF_DEBUG_S(s); LRF_DEBUG_SEP; LRF_DEBUG_ST(v,HEX); LRF_DEBUG_TAB; }
#define LRF_DEBUG_VT(s,v,t)	{ LRF_DEBUG_S(s); LRF_DEBUG_SEP; LRF_DEBUG_ST(v,t); LRF_DEBUG_TAB; }
#define LRF_DEBUG_VNL(s,v)		{ LRF_DEBUG_V(s,v); LRF_DEBUG_NL; }
#define LRF_DEBUG_VTNL(s,v,t)	{ LRF_DEBUG_VT(s,v,t); LRF_DEBUG_NL; }
#define LRF_DEBUG_CV(v)		{ LRF_DEBUG_COMMA; LRF_DEBUG_ST(v,HEX); }

// Print an array of values
#define LRF_DEBUG_VVT(s,vs,l,t)	{ LRF_DEBUG_S(s); LRF_DEBUG_SEP; LRF_DEBUG_OPEN; for(int ii=0;ii<l;ii++){ LRF_DEBUG_ST(vs[ii],t); LRF_DEBUG_SPACE; }; LRF_DEBUG_CLOSE; }
#define LRF_DEBUG_VV(s,vs,l)		LRF_DEBUG_VVT(s,vs,l,HEX)
#define LRF_DEBUG_VVTNL(s,vs,l,t)	{ LRF_DEBUG_VVT(s,vs,l,t); LRF_DEBUG_NL; }
#define LRF_DEBUG_VVNL(s,vs,l)		{ LRF_DEBUG_VV(s,vs,l); LRF_DEBUG_NL; }

// Toggle Debug LED
#define LRF_DEBUG_LED_ON		lrf_leds_setStatus(true)
#define LRF_DEBUG_LED_OFF		lrf_leds_setStatus(false)


inline void LRF_DEBUG_PATTERN(LRFPattern *pattern)
{
	LRF_DEBUG_S("PATT> ");
	LRF_DEBUG_VT("colorA",pattern->colorA,DEC);
	LRF_DEBUG_VT("colorB",pattern->colorB,DEC);
	LRF_DEBUG_VT("transform",pattern->transform,BIN);
	LRF_DEBUG_VTNL("duration",pattern->duration,DEC);
}

inline void LRF_DEBUG_SOUND(LRFSound *sound)
{
	LRF_DEBUG_S("SOUND> ");
	LRF_DEBUG_VT("note",sound->note,DEC);
	LRF_DEBUG_VT("octave",sound->octave,DEC);
	LRF_DEBUG_VT("inton",sound->intonation,DEC);
	LRF_DEBUG_VT("dur",sound->duration,DEC);
	LRF_DEBUG_VTNL("pause",sound->pause,DEC);
}

inline void LRF_DEBUG_EXPRESSION(LRFExpression *expression)
{
	LRF_DEBUG_SNL("EXPRESSION>");
	LRF_DEBUG_PATTERN(&expression->pattern);
	LRF_DEBUG_SOUND(&expression->sounds[0]);
	LRF_DEBUG_SOUND(&expression->sounds[1]);
	LRF_DEBUG_SOUND(&expression->sounds[2]);
	LRF_DEBUG_SOUND(&expression->sounds[3]);
}

inline void LRF_DEBUG_MESSAGE(LRFMessage *msg)
{
	LRF_DEBUG_S("MSG> ");
	LRF_DEBUG_VT("command",msg->cmd,HEX);
	LRF_DEBUG_VT("length",msg->length,DEC);
	LRF_DEBUG_VVTNL("data",msg->data,msg->length,HEX);
}

inline void LRF_DEBUG_EVENT(LRFEvent event)
{
	LRF_DEBUG_VT("Event fired", event, HEX);

	if(event == LRFEvent_None)										LRF_DEBUG_INFO("None")
	else if(event == LRFEvent_RobotPowerUp)				LRF_DEBUG_INFO("RobotPowerUp")
	else if(event == LRFEvent_RobotIsHungry)			LRF_DEBUG_INFO("RobotIsHungry")
	else if(event == LRFEvent_RobotIsBored)				LRF_DEBUG_INFO("RobotIsBored")
	else if(event == LRFEvent_RobotIsSleeping)		LRF_DEBUG_INFO("RobotIsSleeping")
	else if(event == LRFEvent_RobotIsAwake)				LRF_DEBUG_INFO("RobotIsAwake")

	else if(event == LRFEvent_RobotGotMessage)		LRF_DEBUG_INFO("RobotGotMessage")

	else if(event == LRFEvent_Tap)								LRF_DEBUG_INFO("Tap")
	else if(event == LRFEvent_Tickle)							LRF_DEBUG_INFO("Tickle")
	else if(event == LRFEvent_Hug)								LRF_DEBUG_INFO("Hug")

	else if(event == LRFEvent_TapLeft)						LRF_DEBUG_INFO("TapLeft")
	else if(event == LRFEvent_TickleLeft)					LRF_DEBUG_INFO("TickleLeft")
	else if(event == LRFEvent_HugLeft)						LRF_DEBUG_INFO("HugLeft")

	else if(event == LRFEvent_TapRight)						LRF_DEBUG_INFO("TapRight")
	else if(event == LRFEvent_TickleRight)				LRF_DEBUG_INFO("TickleRight")
	else if(event == LRFEvent_HugRight)						LRF_DEBUG_INFO("HugRight")

	else if(event == LRFEvent_LongHug)						LRF_DEBUG_INFO("LongHug")
	else if(event == LRFEvent_LongHugLeft)				LRF_DEBUG_INFO("LongHugLeft")
	else if(event == LRFEvent_LongHugRight)				LRF_DEBUG_INFO("LongHugRight")

	else if(event == LRFEvent_Squeeze)						LRF_DEBUG_INFO("Squeeze")
	else if(event == LRFEvent_LongSqueeze)				LRF_DEBUG_INFO("LongSqueeze")
	else if(event == LRFEvent_SuperSqueeze)				LRF_DEBUG_INFO("SuperSqueeze")
	else if(event == LRFEvent_LongSuperSqueeze)		LRF_DEBUG_INFO("LongSuperSqueeze")

	else if(event == LRFEvent_LightNormal)				LRF_DEBUG_INFO("LightNormal")
	else if(event == LRFEvent_LightTooDark)				LRF_DEBUG_INFO("LightTooDark")
	else if(event == LRFEvent_LightTooBright)			LRF_DEBUG_INFO("LightTooBright")
	else if(event == LRFEvent_LightTooDarkTooLong) 		LRF_DEBUG_INFO("LightTooDarkTooLong")
	else if(event == LRFEvent_LightTooBrightTooLong)	LRF_DEBUG_INFO("LightTooBrightTooLong")

	else if(event == LRFEvent_SoundNormal)				LRF_DEBUG_INFO("SoundNormal")
	else if(event == LRFEvent_SoundTooLoud)				LRF_DEBUG_INFO("SoundTooLoud")
	else if(event == LRFEvent_SoundLikeTalking)		LRF_DEBUG_INFO("SoundLikeTalking")
	else if(event == LRFEvent_SoundLikeNoTalking)	LRF_DEBUG_INFO("SoundLikeNoTalking")

	else if(event == LRFEvent_TiltNone)						LRF_DEBUG_INFO("TiltNone")
	else if(event == LRFEvent_TiltLeft)						LRF_DEBUG_INFO("TiltLeft")
	else if(event == LRFEvent_TiltRight)					LRF_DEBUG_INFO("TiltRight")
	else if(event == LRFEvent_TiltForward)				LRF_DEBUG_INFO("TiltForward")
	else if(event == LRFEvent_TiltBack)						LRF_DEBUG_INFO("TiltBack")
	else if(event == LRFEvent_LayLeft)						LRF_DEBUG_INFO("LayLeft")
	else if(event == LRFEvent_LayRight)						LRF_DEBUG_INFO("LayRight")
	else if(event == LRFEvent_LayForward)					LRF_DEBUG_INFO("LayForward")
	else if(event == LRFEvent_LayBack)						LRF_DEBUG_INFO("LayBack")
	else if(event == LRFEvent_Shake)							LRF_DEBUG_INFO("Shake")

	LRF_DEBUG_NL;
}


#ifdef __cplusplus
}
#endif

#endif
