/*!

 \file lrf_eventHandlers.h
 \brief Contains all the built-in handlers that control how the LRF responds to different events.

 */
#include "../system/data/lrf_expression_data.h"

#ifndef __LRF_CHARACTER_H
#define __LRF_CHARACTER_H

#include "../system/lrf_events.h"
#include "../tricks/lrf_tricks.h"
#include "../lrf_debug.h"

#ifdef __cplusplus
extern "C" {
#endif

#define LRF_CHARACTER_DEBUG_VERBOSE       false

// 1 x 2 byte lookup value
#define LRF_CHARACTER_TEST_ADDRESS        (0x00FE)
#define LRF_CHARACTER_TEST_LENGTH         (2)
#define LRF_CHARACTER_TEST_UNCONFIGURED   (0xFFFF)
#define LRF_CHARACTER_TEST_CONFIGURED     (0x1832)

// 1 x 32 byte name
#define LRF_CHARACTER_NAME_ADDRESS			  (0x0000)
#define LRF_CHARACTER_NAME_SIZE 			    (32)
#define LRF_CHARACTER_NAME_LENGTH			    (32)

// 128 x 1 byte references
#define LRF_CHARACTER_EVENT_MAP_ADDRESS		(0x0100)
#define LRF_CHARACTER_EVENT_MAP_LENGTH		(128)
#define LRF_CHARACTER_EVENT_MAP_COUNT     LRF_EVENT_HANDLER_COUNT

// 16 x 10 byte expressions
#define LRF_CHARACTER_CUSTOM_EXPS_ADDRESS	(0x0200)
#define LRF_CHARACTER_CUSTOM_EXPS_COUNT   (16)
#define LRF_CHARACTER_CUSTOM_EXPS_LENGTH	(160)
//#define LRF_CHARACTER_CUSTOM_EXPS_LENGTH	(LRF_CHARACTER_CUSTOM_EXPS_COUNT * LRF_EXPRESSION_SIZE)

// 16 x 32 byte names
#define LRF_CHARACTER_CUSTOM_NAMES_SIZE 	  (32)		// 16 x 10 byte expressions
#define LRF_CHARACTER_CUSTOM_NAMES_ADDRESS  (0x0300)
#define LRF_CHARACTER_CUSTOM_NAMES_LENGTH	  (LRF_CHARACTER_CUSTOM_EXPS_COUNT * LRF_CHARACTER_CUSTOM_NAMES_SIZE)



typedef struct {
  bool                enabled;
  // LRFCharacterEyesMode eyesMode;
  LRFDebugLevel       debugLevel;
  bool                overwrite;
  LRFColor            color;
  LRFTrickSketch_t    trick;
  char                name[LRF_CHARACTER_NAME_LENGTH];
  LRFExpressionName		eventMap[LRF_CHARACTER_EVENT_MAP_COUNT];
  LRFExpression				customExpressions[LRF_CHARACTER_CUSTOM_EXPS_COUNT];
} lrf_character_t;

extern lrf_character_t lrf_character;

void lrf_character_init(void);
void lrf_character_test(void);

void lrf_character_process(void);

// Writing to the Face
bool lrf_character_write_everything(void);
bool lrf_character_write_configFlag(void);
bool lrf_character_write_name(void);
bool lrf_character_write_eventMap(LRFEvent event=LRFEvent_None);
bool lrf_character_write_customExpressions(LRFExpressionName preset=LRFExpressionName_Null);

// Character Name
void lrf_character_setName(char *name, bool sync=true);
void lrf_character_getName(char *name, int maxLength=LRF_CHARACTER_NAME_SIZE);

// Event Map
void lrf_character_getEventMapping(LRFEvent event, LRFExpressionName *name);
void lrf_character_setEventMapping(LRFEvent event, LRFExpressionName name, bool sync=true);

// Expressions
bool lrf_character_isDefaultExpression(LRFExpressionName name);
bool lrf_character_isCustomExpression(LRFExpressionName name);

// Expression Names
void lrf_character_getCustomExpressionName(LRFExpressionName name, char *str, int maxLength=LRF_CHARACTER_CUSTOM_NAMES_SIZE);
void lrf_character_setCustomExpressionName(LRFExpressionName name, char *str, int length=LRF_CHARACTER_CUSTOM_NAMES_SIZE);


void lrf_character_setCustomExpression(LRFExpressionName name, LRFExpression *expression, bool sync=true);

void lrf_character_getDefaultExpression(LRFExpressionName name, LRFExpression *expression);
void lrf_character_getCustomExpression(LRFExpressionName name, LRFExpression *expression);

void lrf_character_getExpressionPreset(LRFExpressionName name, LRFExpression *expression);
void lrf_character_getExpressionPresetPattern(LRFExpressionName name, LRFPattern *pattern);
void lrf_character_getExpressionPresetSound(LRFExpressionName name, LRFSound *sound, uint8_t soundIdx);
void lrf_character_setExpressionPreset(LRFExpressionName name, LRFExpression *expression, bool sync=true);
void lrf_character_setExpressionPresetPattern(LRFExpressionName name, LRFPattern *pattern, bool sync=true);
void lrf_character_setExpressionPresetSound(LRFExpressionName name, LRFSound *sound, uint8_t soundIdx, bool sync=true);


#ifdef __cplusplus
}
#endif

#endif
